#' Helper function for document-wise computation of ranked retrieval scores
#'
#' Helper function for document-wise computation of ranked retrieval scores
#' DCG, NDCG and LRAP.
#' Implemented as in Annif
#' \url{https://github.com/NatLibFi/Annif/blob/master/annif/eval.py}.

#' Reference implementation of DCG to test against.
#'
#' @param gold_vs_pred A data.frame as generated by \code{create_comparison}.
#' @param limit An integer cutoff value for DCG@@N.
#'
#' @return The numeric value of DCG.
dcg_score <- function(gold_vs_pred, limit = NULL) {
  # return the discounted cumulative gain (DCG) score for the selected
  #  labels vs. relevant labels

  gold_vs_pred <- gold_vs_pred |> dplyr::arrange(dplyr::desc(score))
  n_pred <- sum(!is.na(gold_vs_pred$score))
  if (!is.null(limit)) n_pred <- min(limit, n_pred)

  gain <- gold_vs_pred$gold[1:n_pred]
  discount <- log2(1:n_pred + 1)

  return(sum(gain / discount))
}

##########################################################################

#' Helper function for document-wise computation of ranked retrieval scores
#'
#' Helper function for document-wise computation of ranked retrieval scores
#' DCG, NDCG and LRAP.
#' Implemented as in Annif
#' \url{https://github.com/NatLibFi/Annif/blob/master/annif/eval.py}.

#' Reference implementation for NDCG to test against.
#'
#' @param gold_vs_pred A data.frame as generated by \code{create_comparison}.
#' @param limit An integer cutoff value for NDCG@@N.
#'
#' @return The numeric value of NDCG.
ndcg_score <- function(gold_vs_pred, limit = NULL) {
  # return the normalised discounted cumulative gain (NDCG) score for the
  #  selected labels vs. relevant labels

  scores <- c()

  for (DOC_ID in unique(gold_vs_pred$doc_id)) {
    temp <- gold_vs_pred |> dplyr::filter(doc_id == DOC_ID)

    idcg <- dcg_score(
      temp |> dplyr::mutate(score = gold), limit
    )
    dcg <- dcg_score(
      temp, limit
    )

    if (idcg > 0) {
      scores <- c(scores, dcg / idcg)
    } else {
      scores <- c(scores, 1.0)
    }
  }

  return(mean(scores, na.rm = TRUE))
}

#' Helper function for document-wise computation of ranked retrieval scores
#'
#' Helper function for document-wise computation of ranked retrieval scores
#' DCG, NDCG and LRAP.
#' Implemented as in Annif
#' \url{https://github.com/NatLibFi/Annif/blob/master/annif/eval.py}.

#' Reference implementation for Label Ranking Average Precision.
#'
#' @param gold_vs_pred A data.frame as generated by \code{create_comparison}.
#'
#' @return The numeric value of LRAP.
lrap_score <- function(gold_vs_pred) {
  # compute ranking-based average precision

  out <- c()

  for (DOC_ID in unique(gold_vs_pred$doc_id)) {
    temp <- gold_vs_pred |>
      dplyr::filter(doc_id == DOC_ID) |>
      dplyr::mutate(score = tidyr::replace_na(score, 0))

    relevant <- which(temp$gold)

    if (length(relevant) == 0) {
      aux <- 1.0
    } else {
      scores_i <- -temp$score
      rank <- rank(scores_i, ties.method = "max")[relevant]
      l <- rank(scores_i[relevant], ties.method = "max")

      # manual correction for false negatives: l / rank = #TP / n_samples ~ 0
      aux <- mean((scores_i[relevant] < 0) * l / rank)
    }

    out <- c(out, aux)
  }

  mean(out, na.rm = TRUE)
}
