#' Coefficient matrices of the lagged endogenous variables
#'
#' Returns the estimated coefficient matrices of the lagged endogenous
#' variables as a list of \eqn{(K \times K)} matrices.
#'
#' Consider VAR(p) model:
#' \deqn{\mathbf{y}_t = \mathbf{A}_1 \mathbf{y}_{t-1} + ... + \mathbf{A}_p
#' \mathbf{y}_{t-p} + \mathbf{C} \mathbf{d}_t + \mathbf{e}_t.}
#' The function returns the \eqn{(K \times K)} matrices
#' \eqn{\mathbf{A}_1, ..., \mathbf{A}_p} as a list object.
#'
#' This function modifies \code{vars::Acoef()} for the class "varshrinkest",
#' preventing redundant copying of data matrix objects.
#'
#' @param x An object of class "varshrinkeset", generated by VARshrink().
#' @returns A list object with K-by-K VAR coefficient matrices
#' \eqn{\mathbf{A}_1, ..., \mathbf{A}_p}.
#' @examples
#' data(Canada, package = "vars")
#' y <- diff(Canada)
#' estim <- VARshrink(y, p = 2, type = "const", method = "ridge")
#' Acoef_sh(estim)
#' @seealso \code{\link[vars]{Acoef}}
#' @export
Acoef_sh <- function(x) {

  if (!inherits(x, "varest")) {
    stop("\nPlease provide an object inheriting class 'varest'.\n")
  }
  K <- x$K
  p <- x$p
  A <- Bcoef_sh(x)[, 1:(K * p)]
  As <- list()
  start <- seq(1, p * K, K)
  end <- seq(K, p * K, K)
  for (i in 1:p) {
    As[[i]] <- matrix(A[, start[i]:end[i]], nrow = K, ncol = K)
    rownames(As[[i]]) <- rownames(A)
    colnames(As[[i]]) <- colnames(A[, start[i]:end[i]])
  }
  return(As)
}
