#' Static Visualization of Richness and Trait Maps
#'
#' @description
#' This function is the dedicated plotting tool for outputs from `richness_here()`.
#' It automatically handles single-layer rasters (e.g., species richness) and
#' multi-layer rasters (e.g., multiple biological traits or flags), creating
#' a standardized visual using **ggplot2**.
#'
#' @param raster (SpatRaster) A raster object generated by `richness_here()`.
#' @param low_color (character) color for the lowest values. Default is "blue".
#' @param mid_color (character) color for the midpoint. Default is "yellow".
#' @param high_color (character) color for the highest values. Default is "red".
#' @param alpha (numeric) transparency of the grid (0-1). Default is 0.8.
#' @param continent (SpatVector) optional polygon layer for boundaries.
#' @param continent_fill (character) fill color for continents. Default is "gray70".
#' @param continent_linewidth (numeric) line width for continent boundaries.
#' Default is 0.3.
#' @param continent_border (character) color of the continent polygon borders.
#' Default is "white".
#' @param ocean_fill (character) background color for the ocean. Default is "aliceblue".
#' @param extension (SpatExtent or numeric) optional map extent.
#' @param theme_plot (theme) a `ggplot2` theme object.
#' @param ... other arguments passed to `ggplot2::theme()`.
#'
#' @returns A ggplot object.
#'
#' @importFrom ggplot2 ggplot aes geom_tile scale_fill_gradient2 coord_sf xlab ylab theme element_rect labs
#' @importFrom terra as.data.frame crop unwrap ext nlyr
#' @importFrom sf st_as_sf
#'
#' @export
#'
#' @examples
#' # Load example data
#' data("occ_flagged", package = "RuHere")
#'
#' # Simple richness map
#' r_records <- richness_here(occ_flagged, summary = "records", res = 2)
#' ggrid_here(r_records)
#'
#' # Density of specific flags
#' # Let's see where 'florabr' flags are concentrated
#' r_flags <- richness_here(occ_flagged, summary = "records",
#'                          field = "florabr_flag",
#'                          field_name = "Records flagged by florabr",
#'                          fun = function(x, ...) sum(!x, na.rm = TRUE),
#'                          res = 2)
#' ggrid_here(r_flags)
#'
ggrid_here <- function(raster, low_color = "blue", mid_color = "yellow",
                       high_color = "red", alpha = 0.8, continent = NULL,
                       continent_fill = "gray70", continent_linewidth = 0.3,
                       continent_border = "white", ocean_fill = "aliceblue",
                       extension = NULL,
                       theme_plot = ggplot2::theme_minimal(),
                       ...) {

  if (!inherits(raster, "SpatRaster")) {
    stop("'raster' must be a SpatRaster object from richness_here().", call. = FALSE)
  }

  if (is.null(continent)) {
    continent <- terra::unwrap(getExportedValue("RuHere", "world"))
    continent <- sf::st_as_sf(continent)
  }

  if (is.null(extension)) {
    extension <- as.vector(terra::ext(raster))
  }

  grid_cropped <- terra::crop(raster, terra::ext(extension))

  # Get layer name
  layer_name <- names(raster)

  if(layer_name == "n_records"){
    layer_name <- "Number of records"
  } else if(layer_name == "richness"){
    layer_name <- "Number of species"
  }

  df_layer <- terra::as.data.frame(grid_cropped, xy = TRUE, na.rm = TRUE)
  colnames(df_layer)[3] <- "Value"

  # Get min and max
  min_val <- min(df_layer$Value, na.rm = TRUE)
  max_val <- max(df_layer$Value, na.rm = TRUE)
  mid_val <- ceiling(max(df_layer$Value, na.rm = TRUE)/2)

  p_sub <- ggplot2::ggplot() +
    ggplot2::geom_sf(data = continent, fill = continent_fill,
                     linewidth = continent_linewidth, col = continent_border) +
    ggplot2::geom_tile(data = df_layer,
                       ggplot2::aes(x = .data$x, y = .data$y, fill = .data$Value),
                       alpha = alpha) +
    ggplot2::scale_fill_gradient2(name = layer_name,
                                  low = low_color, mid = mid_color, high = high_color,
                                  midpoint = mid_val,
                                  limits = c(min_val, max_val),
                                  breaks = c(min_val, mid_val, max_val)) +
    ggplot2::coord_sf(xlim = c(extension[1], extension[2]),
                      ylim = c(extension[3], extension[4]), expand = FALSE) +
    ggplot2::labs(x = "Longitude", y = "Latitude") +
    theme_plot +
    ggplot2::theme(panel.background = ggplot2::element_rect(fill = ocean_fill, colour = NA),
                   ...)



    return(p_sub)


}
