# PoissonExpPower.R
#' Poisson Exponential Power Distribution 
#'
#' Provides density, distribution, quantile, random generation,
#' and hazard functions for the Poisson Exponential Power distribution.
#'
#' The Poisson Exponential Power distribution is parameterized by the parameters
#' \eqn{\alpha > 0}, \eqn{\beta > 0}, and \eqn{\lambda > 0}.
#'
#' @details
#' The Poisson Exponential Power distribution has CDF:
#'
#' \deqn{
#' F(x; \alpha, \beta, \lambda) =
#' \quad 1-\frac{1}{\left(1-e^{-\lambda}\right)}\left[1-\exp 
#' \left\{-\lambda \exp \left(1-e^{\beta x^\alpha}\right)\right\}\right]  \quad ;\;x > 0.
#' }
#'
#' where \eqn{\alpha}, \eqn{\beta}, and \eqn{\lambda} are the parameters.
#'
#' The following functions are included:
#' \itemize{
#'   \item \code{dpois.exp.pow()} — Density function
#'   \item \code{ppois.exp.pow()} — Distribution function
#'   \item \code{qpois.exp.pow()} — Quantile function
#'   \item \code{rpois.exp.pow()} — Random generation
#'   \item \code{hpois.exp.pow()} — Hazard function
#' }
#'
#' @usage
#' dpois.exp.pow(x, alpha, beta, lambda, log = FALSE)
#' ppois.exp.pow(q, alpha, beta, lambda, lower.tail = TRUE, log.p = FALSE)
#' qpois.exp.pow(p, alpha, beta, lambda, lower.tail = TRUE, log.p = FALSE)
#' rpois.exp.pow(n, alpha, beta, lambda)
#' hpois.exp.pow(x, alpha, beta, lambda)
#'
#' @param x,q numeric vector of quantiles (x, q)
#' @param p numeric vector of probabilities (0 < p < 1)
#' @param n number of observations (integer > 0)
#' @param alpha positive numeric parameter
#' @param beta positive numeric parameter
#' @param lambda positive numeric parameter
#' @param log logical; if TRUE, returns log-density
#' @param lower.tail logical; if TRUE (default), probabilities are 
#' \eqn{P[X \le x]} otherwise, \eqn{P[X > x]}.
#' @param log.p logical; if TRUE, probabilities are given as log(p)
#'
#' @return
#' \itemize{
#'   \item \code{dpois.exp.pow}: numeric vector of (log-)densities
#'   \item \code{ppois.exp.pow}: numeric vector of probabilities
#'   \item \code{qpois.exp.pow}: numeric vector of quantiles
#'   \item \code{rpois.exp.pow}: numeric vector of random variates
#'   \item \code{hpois.exp.pow}: numeric vector of hazard values
#' }
#'
#' @references 
#' Joshi, R. K., & Kumar, V. (2020). 
#' Poisson Exponential Power distribution: Properties and Application.
#' \emph{International Journal of Mathematics & Computer Research},  
#' \bold{8(11)}, 2152--2158. \doi{10.47191/ijmcr/v8i11.01}
#'
#' @examples
#' x <- seq(0.1, 2.0, 0.2)
#' dpois.exp.pow(x, 2.0, 0.5, 0.2)
#' ppois.exp.pow(x, 2.0, 0.5, 0.2)
#' qpois.exp.pow(0.5, 2.0, 0.5, 0.2)
#' rpois.exp.pow(10, 2.0, 0.5, 0.2)
#' hpois.exp.pow(x, 2.0, 0.5, 0.2)
#' 
#' # Data
#' x <- stress
#' # ML estimates
#' params = list(alpha=0.6976, beta=0.6395, lambda=7.8045)
#' #P–P (probability–probability) plot
#' pp.plot(x, params = params, pfun = ppois.exp.pow, fit.line=TRUE)
#'
#' #Q-Q (quantile–quantile) plot 
#' qq.plot(x, params = params, qfun = qpois.exp.pow, fit.line=TRUE)
#'
#' # Goodness-of-Fit(GoF) and Model Diagnostics 
#' out <- gofic(x, params = params,
#'              dfun = dpois.exp.pow, pfun=ppois.exp.pow, plot=TRUE)
#' print.gofic(out)
#' 
#' @name PoissonExpPower
#' @aliases dpois.exp.pow ppois.exp.pow qpois.exp.pow rpois.exp.pow hpois.exp.pow
#' @export
NULL

#' @rdname PoissonExpPower
#' @usage NULL
#' @export   
dpois.exp.pow <- function (x, alpha, beta, lambda, log = FALSE)
{
	if (!all(sapply(list(x, alpha, beta, lambda), is.numeric))) {
		stop("All arguments must be numeric.")
	}

	if (any(c(alpha, beta, lambda) <= 0)) {
		stop("Parameters 'alpha', 'beta' and 'lambda' must be positive.")
	}

	pdf <- numeric(length(x))

	valid <- x > 0
	if (any(valid)) {	
	tmp <- 1/(1 - exp(-lambda))
	w <- exp(beta * (x[valid] ^ alpha))
	z <- exp(1- w)
    gx <- alpha * beta * lambda *(x[valid] ^(alpha-1)) * w * z * exp(-lambda*z)	
    pdf[valid] <- tmp * gx 
    }
	if (log) 
       pdf<- log(pdf)
    return(pdf)   
}

#' @rdname PoissonExpPower
#' @usage NULL
#' @export     
ppois.exp.pow <- function(q, alpha, beta, lambda, lower.tail = TRUE, log.p = FALSE)
{
	if (!all(sapply(list(q, alpha, beta, lambda), is.numeric))) {
		stop("All arguments must be numeric.")
	}

	if (any(c(alpha, beta, lambda) <= 0)) {
		stop("Parameters 'alpha', 'beta' and 'lambda' must be positive.")
	}

	cdf <- numeric(length(q))

	valid <- q > 0
	if (any(valid)) {
	tmp <- 1/(1 - exp(-lambda))
	w <- exp(beta * (q[valid] ^ alpha))
	z <- exp(1- w)
    cdf[valid] <- 1.0 - tmp * (1-exp(-lambda * z))
    }
	if(!lower.tail)
        cdf <- 1.0 - cdf
    if(log.p)
        cdf <- log(cdf)
    return(cdf)
}

#' @rdname PoissonExpPower
#' @usage NULL
#' @export    
qpois.exp.pow <- function(p, alpha, beta, lambda, lower.tail = TRUE, log.p = FALSE)
{
 	if (!all(sapply(list(p, alpha, beta, lambda), is.numeric)))
		stop("All arguments must be numeric.")
	if (any(c(alpha, beta, lambda) <= 0))
		stop("Invalid parameters: alpha, 'beta' and lambda must be positive.")
	
	if (log.p) p <- exp(p)
    if (!lower.tail) p <- 1 - p

    if (any(p < 0 | p > 1))
		stop("Probabilities 'p' must lie in [0, 1].")

	qtl <- numeric(length(p))
	
	is_zero <- p == 0
	is_one  <- p == 1
	valid  <- p > 0 & p < 1

	if (any(is_zero)) qtl[is_zero] <- 0
	if (any(is_one))  qtl[is_one]  <- Inf
	
	if (any(valid)) {	
	z <- 1+(1/lambda) * log(1-(1-exp(-lambda))*(1-p[valid]))		
    qtl[valid] <- ((1/beta) *log(1- log(1-z))) ^ (1/alpha)     
	}
    return(qtl)   
} 

#' @rdname PoissonExpPower
#' @usage NULL
#' @export   
rpois.exp.pow <- function(n, alpha, beta, lambda)
{
	if (!is.numeric(n) || length(n) != 1L || n <= 0 || n %% 1 != 0)
		stop("'n' must be a positive integer.") 
	if (!all(sapply(list(alpha, beta, lambda), is.numeric)))
		stop("'alpha', and 'lambda' must be numeric.")
	if (any(c(alpha, beta, lambda) <= 0))
		stop("Invalid parameters: alpha, 'beta', and lambda must be positive.")
  
    u <- runif(n)
    q <- qpois.exp.pow(u, alpha, beta, lambda)
    return(q)
}

#' @rdname PoissonExpPower
#' @usage NULL
#' @export   
hpois.exp.pow <- function (x, alpha, beta, lambda)
{
 	if (!all(sapply(list(x, alpha, beta, lambda), is.numeric)))
		stop("All arguments must be numeric.")
	if (any(c(alpha, beta, lambda, x) <= 0)) 
		stop("All parameters and 'x' must be positive.")
    
    nume <- dpois.exp.pow(x, alpha, beta, lambda)
	surv <- 1.0 - ppois.exp.pow(x, alpha, beta, lambda)
	hrf <- ifelse(surv > .Machine$double.eps, nume / surv, Inf)
    return(hrf)    
} 
