#' Create GGE-style biplot annotated by RSI
#'
#' @param data Data frame with genotype, environment, and response columns.
#' @param y Response variable column.
#' @param gen Genotype column.
#' @param env Environment column.
#' @param rsi_table Optional RSI table from compute_rsi().
#'
#' @return A ggplot object of the biplot.
#' @examples
#' library(NPStability)
#' data(example_data)
#' rsi_results <- compute_rsi(example_data, Yield, Gen, Env)
#' rsi_biplot(example_data, Yield, Gen, Env, rsi_table = rsi_results)
#'
#' @import dplyr
#' @import tidyr
#' @import rlang
#' @import ggplot2
#' @importFrom stats prcomp
#' @export
rsi_biplot <- function(data, y, gen, env, rsi_table = NULL){

  GE <- PC1 <- PC2 <- Env <- stab_cat <- size_var <- Genotype <- NULL

  yq   <- rlang::enquo(y)
  genq <- rlang::enquo(gen)
  envq <- rlang::enquo(env)
  gen_name <- rlang::as_name(genq)

  d <- as_tibble(data) %>%
    dplyr::mutate(
      !!genq := as.factor(!!genq),
      !!envq := as.factor(!!envq)
    )

  ge_means <- d %>%
    dplyr::group_by(!!genq, !!envq) %>%
    dplyr::summarise(GE = mean(!!yq, na.rm = TRUE), .groups = "drop")

  ge_wide <- tidyr::pivot_wider(ge_means, names_from = !!envq, values_from = GE)
  ge_df <- as.data.frame(ge_wide)
  rownames(ge_df) <- ge_df[[gen_name]]
  ge_df[[gen_name]] <- NULL
  ge_mat <- as.matrix(ge_df)

  gge_effects <- sweep(ge_mat, 2, colMeans(ge_mat, na.rm = TRUE), FUN = "-")
  pca <- stats::prcomp(gge_effects, scale. = FALSE)

  gscores <- as.data.frame(pca$x[, 1:2])
  gscores$Genotype <- rownames(gscores)
  colnames(gscores)[1:2] <- c("PC1", "PC2")

  loadings <- as.data.frame(pca$rotation[, 1:2])
  loadings$Env <- rownames(loadings)
  colnames(loadings)[1:2] <- c("PC1", "PC2")

  if (is.null(rsi_table)) rsi_table <- compute_rsi(data, !!yq, !!genq, !!envq)

  plotdf <- merge(gscores, rsi_table, by.x = "Genotype", by.y = gen_name, all.x = TRUE)

  plotdf$stab_cat <- cut(plotdf$rRSI, breaks = c(-Inf, 3, 6, Inf),
                         labels = c("Most stable", "Moderately stable", "Less stable"))
  plotdf$size_var <- max(plotdf$rRSI, na.rm = TRUE) - plotdf$rRSI + 1

  p <- ggplot2::ggplot(plotdf, ggplot2::aes(x = PC1, y = PC2)) +
    ggplot2::geom_hline(yintercept = 0, linetype = "dashed", color = "gray60") +
    ggplot2::geom_vline(xintercept = 0, linetype = "dashed", color = "gray60") +
    ggplot2::geom_segment(
      data = loadings,
      ggplot2::aes(
        x = 0, y = 0,
        xend = PC1 * max(abs(plotdf$PC1), na.rm = TRUE) * 0.8,
        yend = PC2 * max(abs(plotdf$PC2), na.rm = TRUE) * 0.8
      ),
      arrow = grid::arrow(length = grid::unit(0.2, "cm")),
      inherit.aes = FALSE
    ) +
    ggplot2::geom_text(
      data = loadings,
      ggplot2::aes(
        x = PC1 * max(abs(plotdf$PC1), na.rm = TRUE) * 0.85,
        y = PC2 * max(abs(plotdf$PC2), na.rm = TRUE) * 0.85,
        label = Env
      ),
      size = 3,
      inherit.aes = FALSE
    ) +
    ggplot2::geom_point(ggplot2::aes(color = stab_cat, size = size_var)) +
    ggplot2::geom_text(ggplot2::aes(label = Genotype), vjust = -1.1, size = 3) +
    ggplot2::labs(
      title = "GGE-style biplot annotated by RSI",
      subtitle = "Genotypes sized by stability (more stable -> larger point)",
      x = paste0("PC1 (", round(100 * summary(pca)$importance[2, 1], 1), "%)"),
      y = paste0("PC2 (", round(100 * summary(pca)$importance[2, 2], 1), "%)")
    ) +
    ggplot2::theme_minimal() +
    ggplot2::guides(size = FALSE)

  return(p)
}
