#' Calibrate (a, b) for eNAP prior
#'
#' Calibrates the tuning parameters \eqn{(a,b)} of the elastic NAP prior. This function supports both the one external trial setting 
#' and multiple external trials setting:
#' \itemize{
#'   \item \emph{Single external trial}  provide y_C2C1 and s_C2C1 as scalars.
#'   \item \emph{Multiple external trials} provide y_C2C1 and s_C2C1 as vectors of same lengths. by default the cross-trial variance
#'   will be automatically calculated by REML, otherwise please provide the cross-trial variance as input parameter: sigma2_hat
#'}
#'
#' @details
#' \strong{Calibration procedure:}
#' \itemize{
#' \item \emph{Consistency case (\eqn{Z = 0}).} Enforce near-full borrowing at exact
#'   consistency by solving \eqn{w'(Z = 0) = t_1} for \eqn{a}.
#'
#' \item \emph{Strong inconsistency case (\eqn{Z(\delta)=\frac{|\delta|}{\sqrt{s_{E,C_2}+s_{E,C_1}+s_{C_2,C_1}}}).}} Enforce minimal borrowing
#'   at a clinically significant difference by targeting the \emph{updated}
#'   weight \eqn{w'(Z(\delta)) = t_0}, with calibrated a from step 1, solve for \eqn{b}.
#' }
#' For further details, see the original NAP paper by Zhang and et al. (manuscript).
#'
#' @param s_EC2,s_EC1,s_C2C1  Sampling variances for post-SoC change period (E vs. C2), pre-SoC change period of current trial (E vs. C1 trial) and external trial (C2 vs. C1 trial)
#' @param mu0,tau0 Mean and variance for the vague component, by default mu0=0 and tau0=1000.
#' @param delta  Positive scalar; Clinically significant difference on the log-HR scale such that direct and indirect evidence should be considered as strongly inconsistent.
#' @param t1,t0  Positive scalar; Calibration targets at consisntency and strongly inconsistency: \eqn{w'(0)=t1} (near 1; default 0.99), \eqn{w'(\delta)=t0} (near 0; default 0.05).
#' @param clip_a,clip_b Numeric Vector of Legnth 2: Minimum and maximum caps for tuning parameters (a,b), by default clip_a=(-5,0.5) and clip_b=(0,50)
#' @param exact Logical (TRUE/FALSE); If TRUE, require the exact solution for parameter (a,b), which further requires more parameters input
#' @param y_EC1,y_C2C1 Log-HR for pre-SoC change period and external trial, required only if exact=TRUE
#' @param sigma2_hat Positive scalar, required only for multiple external trials setting, leave blank if use default REML estimate, otherwise provide user-specified value
#' @param verbose   Logical; print diagnostics.
#'
#' @return list with \code{a}, \code{b}, \code{mode} ("FE" or "RE"), and simple \code{check} summary.
#' 
#' @examples
#' s_EC2 <- 0.2^2; s_EC1 <- 0.18^2; s_C2C1 <- 0.18^2
#' tau0 <- 1000
#' 
#' # One external trial setting
#' tune_param_eNAP(
#'   s_EC2,s_EC1,s_C2C1, tau0=1000,
#'   delta=0.5, t1 = 0.999, t0 = 0.05)
#'
#' # Multiple external trials setting
#' s_C2C1=c(0.19^2,0.18^2,0.20^2)
#' y_C2C1=c(-0.5,-0.45,-0.6)
#' tune_param_eNAP(
#'   s_EC2,s_EC1,s_C2C1, tau0=10,
#'   delta=0.5, t1 = 0.999, t0 = 0.05,
#'   exact=TRUE,y_EC1=-0.8,y_C2C1=y_C2C1)
#'
#' @export
tune_param_eNAP <- function(
    s_EC2, s_EC1, s_C2C1,
    tau0=1000,
    delta=0.5, t1 = 0.999, t0 = 0.05,
    clip_a = c(-5, -0.5),
    clip_b = c(0.00001, 50),
    exact = FALSE,
    y_EC1=-0.5,y_C2C1=-0.5, 
    mu0=0,sigma2_hat=NULL,
    verbose = FALSE
){
  # ---- basic checks ----
  req_pos <- function(x,nm) if(!is.finite(x) || x <= 0) stop(sprintf("`%s` must be > 0.", nm), call. = FALSE)
  req_prob <- function(p,nm) if(!is.finite(p) || p <= 0 || p >= 1) stop(sprintf("`%s` must be in (0,1).", nm), call. = FALSE)
  req_pos(s_EC2,"s_EC2"); req_pos(s_EC1,"s_EC1"); req_pos(tau0,"tau0"); req_pos(delta,"delta")
  if (!is.numeric(s_C2C1) || any(!is.finite(s_C2C1)) || any(s_C2C1 <= 0))
    stop("`s_C2C1` must be a numeric vector of positive finite variances.", call. = FALSE)
  if (length(s_C2C1)!=length(y_C2C1) & is.null(sigma2_hat))
    stop("`y_C2C1` must be provided to estimate cross-trial variance using REML.", call. = FALSE)
  req_prob(t1,"t1"); req_prob(t0,"t0")

  logit <- function(p) log(p/(1 - p))
  ilogit <- function(x) 1/(1 + exp(-x))
  
  ## --- Mode selection (one/multiple external trial) ---
  mode <- if (length(s_C2C1) == 1L) "One external trial" else "Multiple external trials"
  if (mode == "Multiple external trials") {
    if (is.null(sigma2_hat)){
      reml <- metafor::rma.uni(yi = y_C2C1, vi = s_C2C1, method = "REML")
      sigma2_hat<-reml$tau2
    } 
    s_C2C1  <- 1 / sum(1 / (s_C2C1 + sigma2_hat))
    y_C2C1 <- as.numeric(reml$beta)  
  }
  V0=s_EC2+tau0
  v_1 <- 1/(1/s_EC1+1/tau0)
  v_2 <- 1/(1/s_C2C1+1/tau0)
  V_ind <- v_1 + v_2
  s_tot=sum(s_EC2,s_EC1,s_C2C1)
  n_eff<-1/s_tot
  Z<-n_eff^(-0.25)* delta/sqrt(s_tot)
  
  # Approximation mode
  approx_a = 0.5*log(V0/s_tot)-logit(t1)
  approx_b = 0.5*log(V0/s_tot)-0.5*Z^2-logit(t0)-approx_a
  approx_b = approx_b/log(Z+1)
  
  # Exact mode
  mu_ind = v_1*y_EC1/s_EC1-(v_2*y_C2C1/s_C2C1)
  y_EC2=y_EC1-y_C2C1
  exact_a = 0.5*log(V0/s_tot)-0.5*((y_EC2-mu_ind)^2/(s_EC2+V_ind)-(y_EC2-mu0)^2/(s_EC2+tau0))-logit(t1)
  exact_b = 0.5*log(V0/s_tot)-0.5*((y_EC2+delta-mu_ind)^2/(s_EC2+V_ind)-(y_EC2+delta-mu0)^2/(s_EC2+tau0))-logit(t0)-exact_a
  exact_b=exact_b/log(Z+1)

  #Choose between
  
  ################## PROGRESS ###################
  if (!exact){
    calib_a = approx_a
    calib_b = approx_b    
    if(abs((exact_a-approx_a)/approx_a) > 0.05 || abs((exact_b-approx_b)/approx_b) > 0.05) {
      warning("Please consider exact solution and provide log-HR for E-C2, E-C1 and C2-C1",call. = TRUE)
      }
  }
  else if(exact){
    calib_a = exact_a
    calib_b = exact_b
  }
  
  # clip to avoid step-like extremes 
  a_pre <- calib_a; b_pre <- calib_b
  a_fin <- min(max(calib_a, clip_a[1]), clip_a[2])
  b_fin <- min(max(calib_b, clip_b[1]), clip_b[2])
  
  clipped_a <- (a_fin != a_pre)
  clipped_b <- (b_fin != b_pre)
    
    # ---- emit guardrail warnings with concrete suggestions ----
    if (isTRUE(clipped_a)) {
      warning(
        paste0(
          "[eNAP calibration] Parameter 'a' hit the cap.",
          "The threshold for mixing weight under consistency may be too extreme. Consider: ",
          "reduce t1 slightly (e.g., 0.999 -> 0.99)."
        ),
        call. = TRUE
      )
    }
    if (isTRUE(clipped_b)) {
      warning(
        paste0(
          "[eNAP calibration] Parameter 'b' hit the cap.",
          "The slope is overly steep. Consider: ",
          "relax the threshold for updated weight under strongly-inconsistency (e.g., t0: 0.01 -> 0.02); and/or",
          "increase the clinically significant difference (e.g. delta: 0.2 -> 0.3)",
          "or consider increasing information (smaller sampling variance s_EC2)."
        ),
        call. = TRUE
      )
    }
  
  
  # achieved weights
  w_prior <- function(x) 1 / (1 + exp(a_fin + b_fin * log(abs(n_eff^(-0.25)*x/sqrt(s_tot)) + 1)))
  check <- c(
    'Mixing weight (consistency)'     = w_prior(0),
    'Updated weight (consistency)'    = post_w(2,a_fin,b_fin,s_EC2,s_EC1,s_C2C1, y_EC2,y_EC1,y_C2C1,tau0),
    't_1'                             = t1,
    'Mixing weight (strongly inconsistency)'    = w_prior(delta),
    'Updated weight (strongly inconsistency)'   = post_w(2,a_fin,b_fin,s_EC2,s_EC1,s_C2C1, y_EC2+delta,y_EC1,y_C2C1,tau0),
    't_0'                                       = t0
  )
  
  
  .fmt <- function(x, digits = 4) {
    if (is.numeric(x)) formatC(x, format = "f", digits = digits) else as.character(x)
  }
  
  check_tbl <- data.frame(
    Metric = c(
      "Setting",
      "Mixing weight at consistency  w(Z=0)   [prior]",
      "Updated weight at consistency w'(Z=0)    [posterior]",
      "t1 (target updated weight at consistency)",
      "Mixing weight at strongly inconsistency  w'(Z=Z(delta))   [prior]",
      "Updated weight at strongly inconsistency w'(Z=Z(delta))    [posterior]",
      "t0 (target updated weight at strongly inconsistency)"
    ),
    Value = c(
      .fmt(mode, 0),
      .fmt(w_prior(0)),
      .fmt(post_w(2, a_fin, b_fin, s_EC2, s_EC1, s_C2C1, y_EC2,         y_EC1, y_C2C1, tau0)),
      .fmt(t1),
      .fmt(w_prior(delta)),
      .fmt(post_w(2, a_fin, b_fin, s_EC2, s_EC1, s_C2C1, y_EC2 + delta, y_EC1, y_C2C1, tau0)),
      .fmt(t0)
    ),
    stringsAsFactors = FALSE
  )
  
  if (isTRUE(verbose)) {
    message(sprintf("Setting:", mode))
    message(sprintf("[%s] a=%.6f, b=%.6f", a_fin, b_fin))
    message(sprintf("Check: w(0)=%.6f (t1=%.6f); w'(delta)=%.6f (t0=%.6f)",
                    check["w_prior"], t1, check["wprime_delta"], t0))
  }
  
  res <- list(
    a       = a_fin,
    b       = b_fin,
    setting = mode,
    delta   = delta,
    t1      = t1,
    t0      = t0,
    meta    = list(
      s_EC2      = s_EC2,
      s_EC1      = s_EC1,
      s_C2C1     = s_C2C1,
      tau0       = tau0,
      mu0        = mu0,
      y_EC1      = y_EC1,
      y_C2C1     = y_C2C1,
      y_EC2_base = y_EC1 - y_C2C1,
      s_tot      = s_tot
    ),
    check   = check
  )
  
  class(res) <- "eNAP_tuning"
  res
}
#' @export
plot.eNAP_tuning <- function(x,
                             xlim = NULL,
                             n = 201,
                             col_post  = "black",
                             lwd_post  = 2,
                             ...) {
  
  tune <- x   # just to reuse your existing code below
  
  # basic sanity
  if (!is.list(tune) || is.null(tune$a) || is.null(tune$b) || is.null(tune$meta)) {
    stop("`x` must be the object returned by tune_param_eNAP().", call. = FALSE)
  }
  
  a     <- tune$a
  b     <- tune$b
  delta <- tune$delta
  
  meta <- tune$meta
  s_EC2      <- meta$s_EC2
  s_EC1      <- meta$s_EC1
  s_C2C1     <- meta$s_C2C1
  tau0       <- meta$tau0
  y_EC1      <- meta$y_EC1
  y_C2C1     <- meta$y_C2C1
  y_EC2_base <- meta$y_EC2_base
  s_tot      <- meta$s_tot
  
  sd_tot <- sqrt(s_tot)
  
  if (is.null(xlim)) {
    span <- if (is.null(delta)) 3 * sd_tot else max(abs(delta), 3 * sd_tot)
    xlim <- c(-span, span)
  }
  
  x_grid <- seq(xlim[1], xlim[2], length.out = max(51L, as.integer(n)))
  Z_grid <- abs(x_grid) / sd_tot
  

  w_post <- vapply(
    x_grid,
    function(xx) {
      y_EC2_x <- y_EC2_base + xx
      post_w(
        w      = 2,
        a      = a,
        b      = b,
        s_EC2  = s_EC2,
        s_EC1  = s_EC1,
        s_C2C1 = s_C2C1,
        y_EC2  = y_EC2_x,
        y_EC1  = y_EC1,
        y_C2C1 = y_C2C1,
        tau0   = tau0
      )
    },
    numeric(1L)
  )
  
  plot(
    x_grid, w_post,
    type = "n",
    xlab = "Observed inconsistency (x)",
    ylab = "Updated weight",
    ylim = c(0, 1),
    ...
  )
  
  lines(x_grid, w_post,  lwd = lwd_post,  col = col_post,  lty = 1)
  
  abline(v = 0, col = "grey47", lty = 2)
  if (!is.null(delta)) {
    abline(v = delta, col = "grey47", lty = 3)
    abline(v = -delta, col = "grey47", lty = 3)
  }
  
  
  invisible(data.frame(
    x       = x_grid,
    Z       = Z_grid,
    w_post  = w_post
  ))
}
