% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/BayesPET.R
\name{convert_median}
\alias{convert_median}
\title{Solve baseline survival parameters by matching the marginal median survival time}
\usage{
convert_median(
  distribution = "Weibull",
  shape,
  median,
  scale,
  cov_type = NULL,
  cov_dist = NULL,
  beta = NULL,
  S = 20000,
  seed.solveparam = 123,
  interval = if (is.null(scale)) c(1e-15, 100) else c(1e-06, 20000),
  tol = .Machine$double.eps^0.25,
  maxiter = 1000
)
}
\arguments{
\item{distribution}{Character string specifying the baseline survival distribution.
Must be either \code{"Weibull"} or \code{"Loglogistic"} (log-logistic).
Defaults to \code{"Weibull"}.}

\item{shape}{Positive numeric scalar. Baseline shape parameter.
For \code{distribution} = \cr \code{"Weibull"}, this corresponds to the Weibull shape
\eqn{\rho}. For \code{distribution} = \code{"Loglogistic"}, this corresponds to the
log-logistic shape \eqn{a}. Set to \code{NULL} to solve for the shape.}

\item{median}{Positive numeric scalar. Median survival time.
If covariates are provided, this is the \emph{marginal} (population) median
\eqn{m} defined by
\deqn{\mathbb{E}_{\boldsymbol{Z}}\{S(m \mid \boldsymbol{Z})\} = 0.5,}
where \eqn{\boldsymbol{Z}} denotes the covariate vector and the expectation is taken
with respect to the covariate distribution implied by
\code{cov_type} and \code{cov_dist}. Set to \code{NULL} to solve for \code{median}.}

\item{scale}{Positive numeric scalar. Baseline scale parameter.
For \code{"Weibull"}, this is the Weibull scale parameter \eqn{\lambda_0}.
For \code{"Loglogistic"}, this is the log-logistic scale \eqn{b}.
Set to \code{NULL} to solve for the scale.}

\item{cov_type}{Character vector specifying the distribution for each component
of the covariate vector \eqn{\boldsymbol{Z} = (Z_1,\dots,Z_p)} used in the
proportional hazards model.
Each element must be \code{"binary"} or \code{"continuous"}.
If \code{NULL} (default), no covariates are used.}

\item{cov_dist}{Numeric vector of the same length as \code{cov_type}, giving
parameters for the covariate-generating distribution of each \eqn{Z_j}:
\itemize{
\item \code{"binary"}: \eqn{Z_j \sim \mathrm{Bernoulli}(p_j)} with
\eqn{p_j = cov\_dist[j]}.
\item \code{"continuous"}: \eqn{Z_j \sim N(0, \sigma_j^2)} with
\eqn{\sigma_j = cov\_dist[j]}.
}
If \code{NULL} (default), no covariates are used.}

\item{beta}{Numeric vector of regression coefficients \eqn{\boldsymbol{\beta}} in the
proportional hazards linear predictor \eqn{\boldsymbol{Z}^\top \boldsymbol{\beta}}.
Must have the same length as \code{cov_type}.
If \code{NULL} (default), no covariates are used.}

\item{S}{Integer. Monte Carlo sample size used to approximate the marginal survival
when covariates are provided. Defaults to \code{20000}.}

\item{seed.solveparam}{Integer random seed for covariate simulation;
if \code{NULL}, the RNG state is not reset. Defaults to \code{123}.}

\item{interval}{Numeric vector of length 2 giving the lower and upper
bounds passed to \code{\link{uniroot}} for the root-finding procedure
used to solve for the unknown parameter (shape, scale, or median).
The interval must bracket the true solution, i.e., the function values
at the two endpoints must have opposite signs.
Defaults to \code{c(1e-15, 100)} when \code{scale = NULL}
and \code{c(1e-6, 20000)} otherwise.}

\item{tol}{Numeric scalar. Defaults to \code{.Machine$double.eps^0.25}.
Convergence tolerance passed to \code{\link[stats]{uniroot}}.}

\item{maxiter}{Integer giving the maximum number of iterations allowed for
\code{\link[stats]{uniroot}}. An error is raised if the algorithm fails
to converge within this limit. Defaults to \code{1000}.}
}
\value{
A numeric scalar giving the solved parameter:
\itemize{
\item the \code{shape} parameter if \code{shape = NULL};
\item the baseline \code{scale} parameter if \code{scale = NULL};
\item the \code{median} survival time if \code{median = NULL}.
}
}
\description{
In practice, it is sometimes more convenient to work with a marginal
(population-level) median survival time. This function solves for exactly
one unknown baseline quantity - \code{shape}, \code{scale}, or \code{median} -
in a proportional hazards (PH) survival model with either a Weibull or
log-logistic baseline distribution, given the other two.

When covariates are provided, \code{median} is interpreted as the marginal
(population-level) median survival time defined implicitly by
\deqn{\mathbb{E}_{\boldsymbol{Z}}\{S(\mathrm{median}\mid \boldsymbol{Z})\} = 0.5}
and the unknown quantity is obtained by Monte Carlo integration over the
covariate distribution.
}
\details{
The proportional hazards (PH) model assumes that, conditional on a covariate
vector \eqn{\boldsymbol{Z}}, the hazard function satisfies
\deqn{h(t \mid \boldsymbol{Z}) = h_0(t)\exp(\boldsymbol{Z}^\top \boldsymbol{\beta})}
for \eqn{t \ge 0}, where \eqn{h_0(t)} is the baseline hazard and
\eqn{\boldsymbol{\beta}} is a vector of regression coefficients.

The baseline model is specified by \code{distribution} through the baseline
survival function \eqn{S_0(t)}:

\itemize{
\item \code{"Weibull"}: baseline survival is
\deqn{S_0(t) = \exp\{-\lambda_0 t^{\rho}\}}
for \eqn{t \ge 0}, where \eqn{\rho > 0} is the \code{shape} parameter and
\eqn{\lambda_0 > 0} is the \code{scale} parameter. An equivalent
representation used by several standard implementations is
\deqn{S_0(t) = \exp\{-(t/\sigma_0)^{\rho}\}}
for \eqn{t \ge 0}, where \eqn{\sigma_0 > 0} is a reparameterization satisfying
\eqn{\lambda_0 = \sigma_0^{-\rho}}. This latter form is used by
\code{\link[stats]{dweibull}},
\code{\link[eha]{phreg}},
\code{\link[flexsurv]{flexsurvreg}}, and \code{\link[rstan:stan_model]{rstan}}.

\item \code{"Loglogistic"}: baseline survival is
\deqn{S_0(t) = \{1 + (t/b)^{a}\}^{-1}}
for \eqn{t \ge 0}, where \eqn{a>0} is the \code{shape} parameter
and \eqn{b>0} is the \code{scale} parameter.
}
Under the PH assumption, the conditional survival satisfies
\deqn{S(t \mid \boldsymbol{Z}) = S_0(t)^{\exp(\boldsymbol{Z}^\top \boldsymbol{\beta})}.}

When no covariates are supplied (\code{cov_type = NULL}), the median survival time
corresponds to the baseline median and closed-form
solutions are available for some distributions.

When covariates are supplied, the median survival time \eqn{m} is interpreted as
the marginal (population) median defined implicitly by
\deqn{\mathbb{E}_{\boldsymbol{Z}}\{S(m \mid \boldsymbol{Z})\} = 0.5,}
where the expectation is taken with respect to the covariate distribution implied
by \code{cov_type} and \code{cov_dist}. This expectation is approximated using
Monte Carlo simulation with \code{S} draws, and the unknown parameter is obtained
by numerical root finding via \code{\link[stats]{uniroot}}.
}
\examples{
# Weibull: convert a desired median to the corresponding scale parameter
## No covariates
convert_median(
  distribution = "Weibull",
  shape = 3,
  median = 5,
  scale = NULL,
  seed = 123
)
## With covariates
convert_median(
  distribution = "Weibull",
  shape = 3,
  median = 5,
  scale = NULL,
  cov_type = c("binary", "continuous"),
  cov_dist = c(0.5, 1),
  beta = c(1, 0.5),
  seed = 123
)

# Log-logistic: convert median to scale when covariates enter the model
convert_median(
  distribution = "Loglogistic",
  shape = 1.5,
  median = 3,
  scale = NULL,
  cov_type = c("binary", "continuous"),
  cov_dist = c(0.5, 1),
  beta = c(1, 0.5),
  seed = 123
)

}
\seealso{
\code{\link[stats]{dweibull}}
\code{\link[flexsurv]{rllogis}}
}
