/*
 * Decompiled with CFR 0.152.
 */
package org.apache.paimon.memory;

import java.io.DataInput;
import java.io.DataOutput;
import java.io.IOException;
import java.nio.BufferOverflowException;
import java.nio.BufferUnderflowException;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.nio.ReadOnlyBufferException;
import java.util.Objects;
import javax.annotation.Nullable;
import org.apache.paimon.annotation.Public;
import org.apache.paimon.memory.MemoryUtils;
import sun.misc.Unsafe;

@Public
public final class MemorySegment {
    public static final Unsafe UNSAFE = MemoryUtils.UNSAFE;
    public static final long BYTE_ARRAY_BASE_OFFSET = UNSAFE.arrayBaseOffset(byte[].class);
    public static final boolean LITTLE_ENDIAN = ByteOrder.nativeOrder() == ByteOrder.LITTLE_ENDIAN;
    @Nullable
    private byte[] heapMemory;
    @Nullable
    private ByteBuffer offHeapBuffer;
    private long address;
    private final int size;

    private MemorySegment(@Nullable byte[] heapMemory, @Nullable ByteBuffer offHeapBuffer, long address, int size) {
        this.heapMemory = heapMemory;
        this.offHeapBuffer = offHeapBuffer;
        this.address = address;
        this.size = size;
    }

    public static MemorySegment wrap(byte[] buffer) {
        return new MemorySegment(buffer, null, BYTE_ARRAY_BASE_OFFSET, buffer.length);
    }

    public static MemorySegment wrapOffHeapMemory(ByteBuffer buffer) {
        return new MemorySegment(null, buffer, MemoryUtils.getByteBufferAddress(buffer), buffer.capacity());
    }

    public static MemorySegment allocateHeapMemory(int size) {
        return MemorySegment.wrap(new byte[size]);
    }

    public static MemorySegment allocateOffHeapMemory(int size) {
        return MemorySegment.wrapOffHeapMemory(ByteBuffer.allocateDirect(size));
    }

    public int size() {
        return this.size;
    }

    public boolean isOffHeap() {
        return this.heapMemory == null;
    }

    public byte[] getArray() {
        if (this.heapMemory != null) {
            return this.heapMemory;
        }
        throw new IllegalStateException("Memory segment does not represent heap memory");
    }

    public ByteBuffer wrap(int offset, int length) {
        return this.wrapInternal(offset, length);
    }

    private ByteBuffer wrapInternal(int offset, int length) {
        if (this.heapMemory != null) {
            return ByteBuffer.wrap(this.heapMemory, offset, length);
        }
        try {
            ByteBuffer wrapper = Objects.requireNonNull(this.offHeapBuffer).duplicate();
            wrapper.limit(offset + length);
            wrapper.position(offset);
            return wrapper;
        }
        catch (IllegalArgumentException e) {
            throw new IndexOutOfBoundsException();
        }
    }

    public byte get(int index) {
        return UNSAFE.getByte(this.heapMemory, this.address + (long)index);
    }

    public void put(int index, byte b) {
        UNSAFE.putByte(this.heapMemory, this.address + (long)index, b);
    }

    public void get(int index, byte[] dst) {
        this.get(index, dst, 0, dst.length);
    }

    public void put(int index, byte[] src) {
        this.put(index, src, 0, src.length);
    }

    public void get(int index, byte[] dst, int offset, int length) {
        if ((offset | length | offset + length | dst.length - (offset + length)) < 0) {
            throw new IndexOutOfBoundsException();
        }
        UNSAFE.copyMemory(this.heapMemory, this.address + (long)index, dst, BYTE_ARRAY_BASE_OFFSET + (long)offset, length);
    }

    public void put(int index, byte[] src, int offset, int length) {
        if ((offset | length | offset + length | src.length - (offset + length)) < 0) {
            throw new IndexOutOfBoundsException();
        }
        UNSAFE.copyMemory(src, BYTE_ARRAY_BASE_OFFSET + (long)offset, this.heapMemory, this.address + (long)index, length);
    }

    public boolean getBoolean(int index) {
        return this.get(index) != 0;
    }

    public void putBoolean(int index, boolean value) {
        this.put(index, (byte)(value ? 1 : 0));
    }

    public char getChar(int index) {
        return UNSAFE.getChar(this.heapMemory, this.address + (long)index);
    }

    public char getCharLittleEndian(int index) {
        if (LITTLE_ENDIAN) {
            return this.getChar(index);
        }
        return Character.reverseBytes(this.getChar(index));
    }

    public char getCharBigEndian(int index) {
        if (LITTLE_ENDIAN) {
            return Character.reverseBytes(this.getChar(index));
        }
        return this.getChar(index);
    }

    public void putChar(int index, char value) {
        UNSAFE.putChar(this.heapMemory, this.address + (long)index, value);
    }

    public void putCharLittleEndian(int index, char value) {
        if (LITTLE_ENDIAN) {
            this.putChar(index, value);
        } else {
            this.putChar(index, Character.reverseBytes(value));
        }
    }

    public void putCharBigEndian(int index, char value) {
        if (LITTLE_ENDIAN) {
            this.putChar(index, Character.reverseBytes(value));
        } else {
            this.putChar(index, value);
        }
    }

    public short getShort(int index) {
        return UNSAFE.getShort(this.heapMemory, this.address + (long)index);
    }

    public short getShortLittleEndian(int index) {
        if (LITTLE_ENDIAN) {
            return this.getShort(index);
        }
        return Short.reverseBytes(this.getShort(index));
    }

    public short getShortBigEndian(int index) {
        if (LITTLE_ENDIAN) {
            return Short.reverseBytes(this.getShort(index));
        }
        return this.getShort(index);
    }

    public void putShort(int index, short value) {
        UNSAFE.putShort(this.heapMemory, this.address + (long)index, value);
    }

    public void putShortLittleEndian(int index, short value) {
        if (LITTLE_ENDIAN) {
            this.putShort(index, value);
        } else {
            this.putShort(index, Short.reverseBytes(value));
        }
    }

    public void putShortBigEndian(int index, short value) {
        if (LITTLE_ENDIAN) {
            this.putShort(index, Short.reverseBytes(value));
        } else {
            this.putShort(index, value);
        }
    }

    public int getInt(int index) {
        return UNSAFE.getInt(this.heapMemory, this.address + (long)index);
    }

    public int getIntLittleEndian(int index) {
        if (LITTLE_ENDIAN) {
            return this.getInt(index);
        }
        return Integer.reverseBytes(this.getInt(index));
    }

    public int getIntBigEndian(int index) {
        if (LITTLE_ENDIAN) {
            return Integer.reverseBytes(this.getInt(index));
        }
        return this.getInt(index);
    }

    public void putInt(int index, int value) {
        UNSAFE.putInt(this.heapMemory, this.address + (long)index, value);
    }

    public void putIntLittleEndian(int index, int value) {
        if (LITTLE_ENDIAN) {
            this.putInt(index, value);
        } else {
            this.putInt(index, Integer.reverseBytes(value));
        }
    }

    public void putIntBigEndian(int index, int value) {
        if (LITTLE_ENDIAN) {
            this.putInt(index, Integer.reverseBytes(value));
        } else {
            this.putInt(index, value);
        }
    }

    public long getLong(int index) {
        return UNSAFE.getLong(this.heapMemory, this.address + (long)index);
    }

    public long getLongLittleEndian(int index) {
        if (LITTLE_ENDIAN) {
            return this.getLong(index);
        }
        return Long.reverseBytes(this.getLong(index));
    }

    public long getLongBigEndian(int index) {
        if (LITTLE_ENDIAN) {
            return Long.reverseBytes(this.getLong(index));
        }
        return this.getLong(index);
    }

    public void putLong(int index, long value) {
        UNSAFE.putLong(this.heapMemory, this.address + (long)index, value);
    }

    public void putLongLittleEndian(int index, long value) {
        if (LITTLE_ENDIAN) {
            this.putLong(index, value);
        } else {
            this.putLong(index, Long.reverseBytes(value));
        }
    }

    public void putLongBigEndian(int index, long value) {
        if (LITTLE_ENDIAN) {
            this.putLong(index, Long.reverseBytes(value));
        } else {
            this.putLong(index, value);
        }
    }

    public float getFloat(int index) {
        return Float.intBitsToFloat(this.getInt(index));
    }

    public float getFloatLittleEndian(int index) {
        return Float.intBitsToFloat(this.getIntLittleEndian(index));
    }

    public float getFloatBigEndian(int index) {
        return Float.intBitsToFloat(this.getIntBigEndian(index));
    }

    public void putFloat(int index, float value) {
        this.putInt(index, Float.floatToRawIntBits(value));
    }

    public void putFloatLittleEndian(int index, float value) {
        this.putIntLittleEndian(index, Float.floatToRawIntBits(value));
    }

    public void putFloatBigEndian(int index, float value) {
        this.putIntBigEndian(index, Float.floatToRawIntBits(value));
    }

    public double getDouble(int index) {
        return Double.longBitsToDouble(this.getLong(index));
    }

    public double getDoubleLittleEndian(int index) {
        return Double.longBitsToDouble(this.getLongLittleEndian(index));
    }

    public double getDoubleBigEndian(int index) {
        return Double.longBitsToDouble(this.getLongBigEndian(index));
    }

    public void putDouble(int index, double value) {
        this.putLong(index, Double.doubleToRawLongBits(value));
    }

    public void putDoubleLittleEndian(int index, double value) {
        this.putLongLittleEndian(index, Double.doubleToRawLongBits(value));
    }

    public void putDoubleBigEndian(int index, double value) {
        this.putLongBigEndian(index, Double.doubleToRawLongBits(value));
    }

    public void get(DataOutput out, int offset, int length) throws IOException {
        if (this.heapMemory != null) {
            out.write(this.heapMemory, offset, length);
        } else {
            while (length >= 8) {
                out.writeLong(this.getLongBigEndian(offset));
                offset += 8;
                length -= 8;
            }
            while (length > 0) {
                out.writeByte(this.get(offset));
                ++offset;
                --length;
            }
        }
    }

    public void put(DataInput in, int offset, int length) throws IOException {
        if (this.heapMemory != null) {
            in.readFully(this.heapMemory, offset, length);
        } else {
            while (length >= 8) {
                this.putLongBigEndian(offset, in.readLong());
                offset += 8;
                length -= 8;
            }
            while (length > 0) {
                this.put(offset, in.readByte());
                ++offset;
                --length;
            }
        }
    }

    public void get(int offset, ByteBuffer target, int numBytes) {
        if ((offset | numBytes | offset + numBytes) < 0) {
            throw new IndexOutOfBoundsException();
        }
        if (target.isReadOnly()) {
            throw new ReadOnlyBufferException();
        }
        int targetOffset = target.position();
        int remaining = target.remaining();
        if (remaining < numBytes) {
            throw new BufferOverflowException();
        }
        if (target.isDirect()) {
            long targetPointer = MemoryUtils.getByteBufferAddress(target) + (long)targetOffset;
            long sourcePointer = this.address + (long)offset;
            UNSAFE.copyMemory(this.heapMemory, sourcePointer, null, targetPointer, numBytes);
            target.position(targetOffset + numBytes);
        } else if (target.hasArray()) {
            this.get(offset, target.array(), targetOffset + target.arrayOffset(), numBytes);
            target.position(targetOffset + numBytes);
        } else {
            throw new IllegalArgumentException("The target buffer is not direct, and has no array.");
        }
    }

    public void put(int offset, ByteBuffer source2, int numBytes) {
        if ((offset | numBytes | offset + numBytes) < 0) {
            throw new IndexOutOfBoundsException();
        }
        int sourceOffset = source2.position();
        int remaining = source2.remaining();
        if (remaining < numBytes) {
            throw new BufferUnderflowException();
        }
        if (source2.isDirect()) {
            long sourcePointer = MemoryUtils.getByteBufferAddress(source2) + (long)sourceOffset;
            long targetPointer = this.address + (long)offset;
            UNSAFE.copyMemory(null, sourcePointer, this.heapMemory, targetPointer, numBytes);
            source2.position(sourceOffset + numBytes);
        } else if (source2.hasArray()) {
            this.put(offset, source2.array(), sourceOffset + source2.arrayOffset(), numBytes);
            source2.position(sourceOffset + numBytes);
        } else {
            for (int i = 0; i < numBytes; ++i) {
                this.put(offset++, source2.get());
            }
        }
    }

    public void copyTo(int offset, MemorySegment target, int targetOffset, int numBytes) {
        byte[] thisHeapRef = this.heapMemory;
        byte[] otherHeapRef = target.heapMemory;
        long thisPointer = this.address + (long)offset;
        long otherPointer = target.address + (long)targetOffset;
        UNSAFE.copyMemory(thisHeapRef, thisPointer, otherHeapRef, otherPointer, numBytes);
    }

    public void copyToUnsafe(int offset, Object target, int targetPointer, int numBytes) {
        UNSAFE.copyMemory(this.heapMemory, this.address + (long)offset, target, targetPointer, numBytes);
    }

    public void copyFromUnsafe(int offset, Object source2, int sourcePointer, int numBytes) {
        UNSAFE.copyMemory(source2, sourcePointer, this.heapMemory, this.address + (long)offset, numBytes);
    }

    public int compare(MemorySegment seg2, int offset1, int offset2, int len) {
        while (len >= 8) {
            long l2;
            long l1 = this.getLongBigEndian(offset1);
            if (l1 != (l2 = seg2.getLongBigEndian(offset2))) {
                return l1 < l2 ^ l1 < 0L ^ l2 < 0L ? -1 : 1;
            }
            offset1 += 8;
            offset2 += 8;
            len -= 8;
        }
        while (len > 0) {
            int b2;
            int b1 = this.get(offset1) & 0xFF;
            int cmp = b1 - (b2 = seg2.get(offset2) & 0xFF);
            if (cmp != 0) {
                return cmp;
            }
            ++offset1;
            ++offset2;
            --len;
        }
        return 0;
    }

    public int compare(MemorySegment seg2, int offset1, int offset2, int len1, int len2) {
        int minLength = Math.min(len1, len2);
        int c = this.compare(seg2, offset1, offset2, minLength);
        return c == 0 ? len1 - len2 : c;
    }

    public void swapBytes(byte[] tempBuffer, MemorySegment seg2, int offset1, int offset2, int len) {
        if ((offset1 | offset2 | len | tempBuffer.length - len) >= 0) {
            long thisPos = this.address + (long)offset1;
            long otherPos = seg2.address + (long)offset2;
            UNSAFE.copyMemory(this.heapMemory, thisPos, tempBuffer, BYTE_ARRAY_BASE_OFFSET, len);
            UNSAFE.copyMemory(seg2.heapMemory, otherPos, this.heapMemory, thisPos, len);
            UNSAFE.copyMemory(tempBuffer, BYTE_ARRAY_BASE_OFFSET, seg2.heapMemory, otherPos, len);
            return;
        }
        throw new IndexOutOfBoundsException(String.format("offset1=%d, offset2=%d, len=%d, bufferSize=%d, address1=%d, address2=%d", offset1, offset2, len, tempBuffer.length, this.address, seg2.address));
    }

    public boolean equalTo(MemorySegment seg2, int offset1, int offset2, int length) {
        int i;
        for (i = 0; i <= length - 8; i += 8) {
            if (this.getLong(offset1 + i) == seg2.getLong(offset2 + i)) continue;
            return false;
        }
        while (i < length) {
            if (this.get(offset1 + i) != seg2.get(offset2 + i)) {
                return false;
            }
            ++i;
        }
        return true;
    }

    public byte[] getHeapMemory() {
        return this.heapMemory;
    }
}

