/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hdfs.server.datanode;

import java.io.File;
import java.io.FilenameFilter;
import java.io.IOException;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledThreadPoolExecutor;
import java.util.concurrent.ThreadLocalRandom;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicLong;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.apache.commons.lang.time.FastDateFormat;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.flink.shaded.hadoop2.com.google.common.annotations.VisibleForTesting;
import org.apache.hadoop.classification.InterfaceAudience;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.hdfs.protocol.Block;
import org.apache.hadoop.hdfs.server.datanode.DataNode;
import org.apache.hadoop.hdfs.server.datanode.DatanodeUtil;
import org.apache.hadoop.hdfs.server.datanode.FinalizedReplica;
import org.apache.hadoop.hdfs.server.datanode.fsdataset.FsDatasetSpi;
import org.apache.hadoop.hdfs.server.datanode.fsdataset.FsVolumeSpi;
import org.apache.hadoop.io.IOUtils;
import org.apache.hadoop.util.AutoCloseableLock;
import org.apache.hadoop.util.Daemon;
import org.apache.hadoop.util.StopWatch;
import org.apache.hadoop.util.Time;

@InterfaceAudience.Private
public class DirectoryScanner
implements Runnable {
    private static final Log LOG = LogFactory.getLog(DirectoryScanner.class);
    private static final int MILLIS_PER_SECOND = 1000;
    private static final String START_MESSAGE = "Periodic Directory Tree Verification scan starting at %s with interval of %dms";
    private static final String START_MESSAGE_WITH_THROTTLE = "Periodic Directory Tree Verification scan starting at %s with interval of %dms and throttle limit of %dms/s";
    private final FsDatasetSpi<?> dataset;
    private final ExecutorService reportCompileThreadPool;
    private final ScheduledExecutorService masterThread;
    private final long scanPeriodMsecs;
    private final int throttleLimitMsPerSec;
    private volatile boolean shouldRun = false;
    private boolean retainDiffs = false;
    private final DataNode datanode;
    @VisibleForTesting
    final AtomicLong timeRunningMs = new AtomicLong(0L);
    @VisibleForTesting
    final AtomicLong timeWaitingMs = new AtomicLong(0L);
    @VisibleForTesting
    final ScanInfoPerBlockPool diffs = new ScanInfoPerBlockPool();
    @VisibleForTesting
    final Map<String, Stats> stats = new HashMap<String, Stats>();

    @VisibleForTesting
    void setRetainDiffs(boolean b) {
        this.retainDiffs = b;
    }

    DirectoryScanner(DataNode datanode, FsDatasetSpi<?> dataset, Configuration conf) {
        this.datanode = datanode;
        this.dataset = dataset;
        int interval = conf.getInt("dfs.datanode.directoryscan.interval", 21600);
        this.scanPeriodMsecs = interval * 1000;
        int throttle = conf.getInt("dfs.datanode.directoryscan.throttle.limit.ms.per.sec", 1000);
        if (throttle > 1000 || throttle <= 0) {
            if (throttle > 1000) {
                LOG.error("dfs.datanode.directoryscan.throttle.limit.ms.per.sec set to value above 1000 ms/sec. Assuming default value of 1000");
            } else {
                LOG.error("dfs.datanode.directoryscan.throttle.limit.ms.per.sec set to value below 1 ms/sec. Assuming default value of 1000");
            }
            this.throttleLimitMsPerSec = 1000;
        } else {
            this.throttleLimitMsPerSec = throttle;
        }
        int threads = conf.getInt("dfs.datanode.directoryscan.threads", 1);
        this.reportCompileThreadPool = Executors.newFixedThreadPool(threads, new Daemon.DaemonFactory());
        this.masterThread = new ScheduledThreadPoolExecutor(1, new Daemon.DaemonFactory());
    }

    void start() {
        this.shouldRun = true;
        long offset = ThreadLocalRandom.current().nextInt((int)(this.scanPeriodMsecs / 1000L)) * 1000;
        long firstScanTime = Time.now() + offset;
        String logMsg = this.throttleLimitMsPerSec < 1000 ? String.format(START_MESSAGE_WITH_THROTTLE, FastDateFormat.getInstance().format(firstScanTime), this.scanPeriodMsecs, this.throttleLimitMsPerSec) : String.format(START_MESSAGE, FastDateFormat.getInstance().format(firstScanTime), this.scanPeriodMsecs);
        LOG.info(logMsg);
        this.masterThread.scheduleAtFixedRate(this, offset, this.scanPeriodMsecs, TimeUnit.MILLISECONDS);
    }

    @VisibleForTesting
    boolean getRunStatus() {
        return this.shouldRun;
    }

    private void clear() {
        this.diffs.clear();
        this.stats.clear();
    }

    @Override
    public void run() {
        try {
            if (!this.shouldRun) {
                LOG.warn("this cycle terminating immediately because 'shouldRun' has been deactivated");
                return;
            }
            this.reconcile();
        }
        catch (Exception e) {
            LOG.error("Exception during DirectoryScanner execution - will continue next cycle", e);
        }
        catch (Error er) {
            LOG.error("System Error during DirectoryScanner execution - permanently terminating periodic scanner", er);
            throw er;
        }
    }

    void shutdown() {
        if (!this.shouldRun) {
            LOG.warn("DirectoryScanner: shutdown has been called, but periodic scanner not started");
        } else {
            LOG.warn("DirectoryScanner: shutdown has been called");
        }
        this.shouldRun = false;
        if (this.masterThread != null) {
            this.masterThread.shutdown();
        }
        if (this.reportCompileThreadPool != null) {
            this.reportCompileThreadPool.shutdownNow();
        }
        if (this.masterThread != null) {
            try {
                this.masterThread.awaitTermination(1L, TimeUnit.MINUTES);
            }
            catch (InterruptedException e) {
                LOG.error("interrupted while waiting for masterThread to terminate", e);
            }
        }
        if (this.reportCompileThreadPool != null) {
            try {
                this.reportCompileThreadPool.awaitTermination(1L, TimeUnit.MINUTES);
            }
            catch (InterruptedException e) {
                LOG.error("interrupted while waiting for reportCompileThreadPool to terminate", e);
            }
        }
        if (!this.retainDiffs) {
            this.clear();
        }
    }

    @VisibleForTesting
    void reconcile() throws IOException {
        this.scan();
        for (Map.Entry entry : this.diffs.entrySet()) {
            String bpid = (String)entry.getKey();
            LinkedList diff = (LinkedList)entry.getValue();
            for (ScanInfo info : diff) {
                this.dataset.checkAndUpdate(bpid, info.getBlockId(), info.getBlockFile(), info.getMetaFile(), info.getVolume());
            }
        }
        if (!this.retainDiffs) {
            this.clear();
        }
    }

    private void scan() {
        this.clear();
        Map<String, ScanInfo[]> diskReport = this.getDiskReport();
        try (AutoCloseableLock lock = this.dataset.acquireDatasetLock();){
            for (Map.Entry<String, ScanInfo[]> entry : diskReport.entrySet()) {
                String bpid = entry.getKey();
                ScanInfo[] blockpoolReport = entry.getValue();
                Stats statsRecord = new Stats(bpid);
                this.stats.put(bpid, statsRecord);
                LinkedList<ScanInfo> diffRecord = new LinkedList<ScanInfo>();
                this.diffs.put(bpid, diffRecord);
                statsRecord.totalBlocks = blockpoolReport.length;
                List<FinalizedReplica> bl = this.dataset.getFinalizedBlocks(bpid);
                Collections.sort(bl);
                int d = 0;
                int m = 0;
                while (m < bl.size() && d < blockpoolReport.length) {
                    FinalizedReplica memBlock = bl.get(m);
                    ScanInfo info = blockpoolReport[d];
                    if (info.getBlockId() < memBlock.getBlockId()) {
                        if (!this.dataset.isDeletingBlock(bpid, info.getBlockId())) {
                            ++statsRecord.missingMemoryBlocks;
                            this.addDifference(diffRecord, statsRecord, info);
                        }
                        ++d;
                        continue;
                    }
                    if (info.getBlockId() > memBlock.getBlockId()) {
                        this.addDifference(diffRecord, statsRecord, memBlock.getBlockId(), info.getVolume());
                        ++m;
                        continue;
                    }
                    if (info.getBlockFile() == null) {
                        this.addDifference(diffRecord, statsRecord, info);
                    } else if (info.getGenStamp() != memBlock.getGenerationStamp() || info.getBlockFileLength() != memBlock.getNumBytes()) {
                        ++statsRecord.mismatchBlocks;
                        this.addDifference(diffRecord, statsRecord, info);
                    } else if (info.getBlockFile().compareTo(memBlock.getBlockFile()) != 0) {
                        ++statsRecord.duplicateBlocks;
                        this.addDifference(diffRecord, statsRecord, info);
                    }
                    if (++d < blockpoolReport.length) {
                        ScanInfo nextInfo = blockpoolReport[Math.min(d, blockpoolReport.length - 1)];
                        if (nextInfo.getBlockId() == info.blockId) continue;
                        ++m;
                        continue;
                    }
                    ++m;
                }
                while (m < bl.size()) {
                    FinalizedReplica current = bl.get(m++);
                    this.addDifference(diffRecord, statsRecord, current.getBlockId(), current.getVolume());
                }
                while (d < blockpoolReport.length) {
                    if (!this.dataset.isDeletingBlock(bpid, blockpoolReport[d].getBlockId())) {
                        ++statsRecord.missingMemoryBlocks;
                        this.addDifference(diffRecord, statsRecord, blockpoolReport[d]);
                    }
                    ++d;
                }
                LOG.info(statsRecord.toString());
            }
        }
    }

    private void addDifference(LinkedList<ScanInfo> diffRecord, Stats statsRecord, ScanInfo info) {
        statsRecord.missingMetaFile = statsRecord.missingMetaFile + (info.getMetaFile() == null ? 1L : 0L);
        statsRecord.missingBlockFile = statsRecord.missingBlockFile + (info.getBlockFile() == null ? 1L : 0L);
        diffRecord.add(info);
    }

    private void addDifference(LinkedList<ScanInfo> diffRecord, Stats statsRecord, long blockId, FsVolumeSpi vol) {
        ++statsRecord.missingBlockFile;
        ++statsRecord.missingMetaFile;
        diffRecord.add(new ScanInfo(blockId, null, null, vol));
    }

    private Map<String, ScanInfo[]> getDiskReport() {
        ScanInfoPerBlockPool list = new ScanInfoPerBlockPool();
        ScanInfoPerBlockPool[] dirReports = null;
        try (FsDatasetSpi.FsVolumeReferences volumes = this.dataset.getFsVolumeReferences();){
            dirReports = new ScanInfoPerBlockPool[volumes.size()];
            HashMap<Integer, Future<ScanInfoPerBlockPool>> compilersInProgress = new HashMap<Integer, Future<ScanInfoPerBlockPool>>();
            for (int i = 0; i < volumes.size(); ++i) {
                ReportCompiler reportCompiler = new ReportCompiler(this.datanode, volumes.get(i));
                Future<ScanInfoPerBlockPool> result = this.reportCompileThreadPool.submit(reportCompiler);
                compilersInProgress.put(i, result);
            }
            for (Map.Entry report : compilersInProgress.entrySet()) {
                Integer index = (Integer)report.getKey();
                try {
                    dirReports[index.intValue()] = (ScanInfoPerBlockPool)((Future)report.getValue()).get();
                    if (dirReports[index] != null) continue;
                    dirReports = null;
                    break;
                }
                catch (Exception ex) {
                    FsVolumeSpi fsVolumeSpi = volumes.get(index);
                    LOG.error("Error compiling report for the volume, StorageId: " + fsVolumeSpi.getStorageID(), ex);
                }
            }
        }
        catch (IOException e) {
            LOG.error("Unexpected IOException by closing FsVolumeReference", e);
        }
        if (dirReports != null) {
            for (ScanInfoPerBlockPool report : dirReports) {
                if (report == null) continue;
                list.addAll(report);
            }
        }
        return list.toSortedArrays();
    }

    private static boolean isBlockMetaFile(String blockId, String metaFile) {
        return metaFile.startsWith(blockId) && metaFile.endsWith(".meta");
    }

    private static enum BlockDirFilter implements FilenameFilter
    {
        INSTANCE;


        @Override
        public boolean accept(File dir, String name) {
            return name.startsWith("subdir") || name.startsWith("finalized") || name.startsWith("blk_");
        }
    }

    private class ReportCompiler
    implements Callable<ScanInfoPerBlockPool> {
        private final FsVolumeSpi volume;
        private final DataNode datanode;
        private final StopWatch throttleTimer = new StopWatch();
        private final StopWatch perfTimer = new StopWatch();

        public ReportCompiler(DataNode datanode, FsVolumeSpi volume) {
            this.datanode = datanode;
            this.volume = volume;
        }

        @Override
        public ScanInfoPerBlockPool call() throws IOException {
            String[] bpList = this.volume.getBlockPoolList();
            ScanInfoPerBlockPool result = new ScanInfoPerBlockPool(bpList.length);
            this.perfTimer.start();
            this.throttleTimer.start();
            for (String bpid : bpList) {
                LinkedList<ScanInfo> report = new LinkedList<ScanInfo>();
                File bpFinalizedDir = this.volume.getFinalizedDir(bpid);
                try {
                    result.put(bpid, this.compileReport(this.volume, bpFinalizedDir, bpFinalizedDir, report));
                }
                catch (InterruptedException ex) {
                    result = null;
                    break;
                }
            }
            return result;
        }

        private LinkedList<ScanInfo> compileReport(FsVolumeSpi vol, File bpFinalizedDir, File dir, LinkedList<ScanInfo> report) throws InterruptedException {
            List<String> fileNames;
            this.throttle();
            try {
                fileNames = IOUtils.listDirectory(dir, BlockDirFilter.INSTANCE);
            }
            catch (IOException ioe) {
                LOG.warn("Exception occured while compiling report: ", ioe);
                this.datanode.checkDiskErrorAsync();
                return report;
            }
            Collections.sort(fileNames);
            for (int i = 0; i < fileNames.size(); ++i) {
                File blkMetaFile;
                if (Thread.interrupted()) {
                    throw new InterruptedException();
                }
                File file = new File(dir, fileNames.get(i));
                if (file.isDirectory()) {
                    this.compileReport(vol, bpFinalizedDir, file, report);
                    continue;
                }
                if (!Block.isBlockFilename(file)) {
                    if (!DirectoryScanner.isBlockMetaFile("blk_", file.getName())) continue;
                    long blockId = Block.getBlockId(file.getName());
                    this.verifyFileLocation(file.getParentFile(), bpFinalizedDir, blockId);
                    report.add(new ScanInfo(blockId, null, file, vol));
                    continue;
                }
                File blockFile = file;
                long blockId = Block.filename2id(file.getName());
                File metaFile = null;
                while (i + 1 < fileNames.size() && (blkMetaFile = new File(dir, fileNames.get(i + 1))).isFile() && blkMetaFile.getName().startsWith(blockFile.getName())) {
                    ++i;
                    if (!DirectoryScanner.isBlockMetaFile(blockFile.getName(), blkMetaFile.getName())) continue;
                    metaFile = blkMetaFile;
                    break;
                }
                this.verifyFileLocation(blockFile, bpFinalizedDir, blockId);
                report.add(new ScanInfo(blockId, blockFile, metaFile, vol));
            }
            return report;
        }

        private void verifyFileLocation(File actualBlockFile, File bpFinalizedDir, long blockId) {
            File blockDir = DatanodeUtil.idToBlockDir(bpFinalizedDir, blockId);
            if (actualBlockFile.getParentFile().compareTo(blockDir) != 0) {
                File expBlockFile = new File(blockDir, actualBlockFile.getName());
                LOG.warn("Block: " + blockId + " has to be upgraded to block ID-based layout. " + "Actual block file path: " + actualBlockFile + ", expected block file path: " + expBlockFile);
            }
        }

        private void throttle() throws InterruptedException {
            this.accumulateTimeRunning();
            if (DirectoryScanner.this.throttleLimitMsPerSec < 1000 && this.throttleTimer.now(TimeUnit.MILLISECONDS) > (long)DirectoryScanner.this.throttleLimitMsPerSec) {
                Thread.sleep(1000 - DirectoryScanner.this.throttleLimitMsPerSec);
                this.throttleTimer.reset().start();
            }
            this.accumulateTimeWaiting();
        }

        private void accumulateTimeRunning() {
            DirectoryScanner.this.timeRunningMs.getAndAdd(this.perfTimer.now(TimeUnit.MILLISECONDS));
            this.perfTimer.reset().start();
        }

        private void accumulateTimeWaiting() {
            DirectoryScanner.this.timeWaitingMs.getAndAdd(this.perfTimer.now(TimeUnit.MILLISECONDS));
            this.perfTimer.reset().start();
        }
    }

    static class ScanInfo
    implements Comparable<ScanInfo> {
        private final long blockId;
        private final String blockSuffix;
        private final String metaSuffix;
        private final FsVolumeSpi volume;
        private final long blockFileLength;
        private static final Pattern CONDENSED_PATH_REGEX = Pattern.compile("(?<!^)(\\\\|/){2,}");
        private static final String QUOTED_FILE_SEPARATOR = Matcher.quoteReplacement(File.separator);

        private static String getCondensedPath(String path) {
            return CONDENSED_PATH_REGEX.matcher(path).replaceAll(QUOTED_FILE_SEPARATOR);
        }

        private static String getSuffix(File f, String prefix) {
            String fullPath = ScanInfo.getCondensedPath(f.getAbsolutePath());
            if (fullPath.startsWith(prefix)) {
                return fullPath.substring(prefix.length());
            }
            throw new RuntimeException(prefix + " is not a prefix of " + fullPath);
        }

        ScanInfo(long blockId, File blockFile, File metaFile, FsVolumeSpi vol) {
            this.blockId = blockId;
            String condensedVolPath = vol == null ? null : ScanInfo.getCondensedPath(vol.getBasePath());
            this.blockSuffix = blockFile == null ? null : ScanInfo.getSuffix(blockFile, condensedVolPath);
            long l = this.blockFileLength = blockFile != null ? blockFile.length() : 0L;
            this.metaSuffix = metaFile == null ? null : (blockFile == null ? ScanInfo.getSuffix(metaFile, condensedVolPath) : ScanInfo.getSuffix(metaFile, condensedVolPath + this.blockSuffix));
            this.volume = vol;
        }

        File getBlockFile() {
            return this.blockSuffix == null ? null : new File(this.volume.getBasePath(), this.blockSuffix);
        }

        long getBlockFileLength() {
            return this.blockFileLength;
        }

        File getMetaFile() {
            if (this.metaSuffix == null) {
                return null;
            }
            if (this.blockSuffix == null) {
                return new File(this.volume.getBasePath(), this.metaSuffix);
            }
            return new File(this.volume.getBasePath(), this.blockSuffix + this.metaSuffix);
        }

        long getBlockId() {
            return this.blockId;
        }

        FsVolumeSpi getVolume() {
            return this.volume;
        }

        @Override
        public int compareTo(ScanInfo b) {
            if (this.blockId < b.blockId) {
                return -1;
            }
            if (this.blockId == b.blockId) {
                return 0;
            }
            return 1;
        }

        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (!(o instanceof ScanInfo)) {
                return false;
            }
            return this.blockId == ((ScanInfo)o).blockId;
        }

        public int hashCode() {
            return (int)(this.blockId ^ this.blockId >>> 32);
        }

        public long getGenStamp() {
            return this.metaSuffix != null ? Block.getGenerationStamp(this.getMetaFile().getName()) : 0L;
        }
    }

    static class ScanInfoPerBlockPool
    extends HashMap<String, LinkedList<ScanInfo>> {
        private static final long serialVersionUID = 1L;

        ScanInfoPerBlockPool() {
        }

        ScanInfoPerBlockPool(int sz) {
            super(sz);
        }

        public void addAll(ScanInfoPerBlockPool that) {
            if (that == null) {
                return;
            }
            for (Map.Entry entry : that.entrySet()) {
                String bpid = (String)entry.getKey();
                LinkedList list = (LinkedList)entry.getValue();
                if (this.containsKey(bpid)) {
                    ((LinkedList)this.get(bpid)).addAll(list);
                    continue;
                }
                this.put(bpid, list);
            }
        }

        public Map<String, ScanInfo[]> toSortedArrays() {
            HashMap<String, ScanInfo[]> result = new HashMap<String, ScanInfo[]>(this.size());
            for (Map.Entry entry : this.entrySet()) {
                String bpid = (String)entry.getKey();
                LinkedList list = (LinkedList)entry.getValue();
                Object[] record = list.toArray(new ScanInfo[list.size()]);
                Arrays.sort(record);
                result.put(bpid, (ScanInfo[])record);
            }
            return result;
        }
    }

    @VisibleForTesting
    static class Stats {
        final String bpid;
        long totalBlocks = 0L;
        long missingMetaFile = 0L;
        long missingBlockFile = 0L;
        long missingMemoryBlocks = 0L;
        long mismatchBlocks = 0L;
        long duplicateBlocks = 0L;

        public Stats(String bpid) {
            this.bpid = bpid;
        }

        public String toString() {
            return "BlockPool " + this.bpid + " Total blocks: " + this.totalBlocks + ", missing metadata files:" + this.missingMetaFile + ", missing block files:" + this.missingBlockFile + ", missing blocks in memory:" + this.missingMemoryBlocks + ", mismatched blocks:" + this.mismatchBlocks;
        }
    }
}

