/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.fs.azurebfs.services;

import java.io.IOException;
import java.util.concurrent.TimeUnit;
import org.apache.hadoop.fs.azurebfs.constants.FSOperationType;
import org.apache.hadoop.fs.azurebfs.contracts.exceptions.AzureBlobFileSystemException;
import org.apache.hadoop.fs.azurebfs.services.AbfsClient;
import org.apache.hadoop.fs.azurebfs.services.AbfsRestOperation;
import org.apache.hadoop.fs.azurebfs.utils.TracingContext;
import org.apache.hadoop.io.retry.RetryPolicies;
import org.apache.hadoop.io.retry.RetryPolicy;
import org.apache.hadoop.thirdparty.com.google.common.annotations.VisibleForTesting;
import org.apache.hadoop.thirdparty.com.google.common.util.concurrent.FutureCallback;
import org.apache.hadoop.thirdparty.com.google.common.util.concurrent.ListenableScheduledFuture;
import org.apache.hadoop.thirdparty.org.checkerframework.checker.nullness.qual.Nullable;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class AbfsLease {
    private static final Logger LOG = LoggerFactory.getLogger(AbfsLease.class);
    static final int DEFAULT_LEASE_ACQUIRE_MAX_RETRIES = 7;
    static final int DEFAULT_LEASE_ACQUIRE_RETRY_INTERVAL = 10;
    private final AbfsClient client;
    private final String path;
    private final TracingContext tracingContext;
    private volatile boolean leaseFreed = false;
    private volatile String leaseID = null;
    private volatile Throwable exception = null;
    private volatile int acquireRetryCount = 0;
    private volatile ListenableScheduledFuture<AbfsRestOperation> future = null;

    public AbfsLease(AbfsClient client, String path, TracingContext tracingContext) throws AzureBlobFileSystemException {
        this(client, path, 7, 10, tracingContext);
    }

    @VisibleForTesting
    public AbfsLease(AbfsClient client, String path, int acquireMaxRetries, int acquireRetryInterval, TracingContext tracingContext) throws AzureBlobFileSystemException {
        this.client = client;
        this.path = path;
        this.tracingContext = tracingContext;
        if (client.getNumLeaseThreads() < 1) {
            throw new LeaseException("Lease desired but no lease threads configured, set fs.azure.lease.threads");
        }
        RetryPolicy retryPolicy = RetryPolicies.retryUpToMaximumCountWithFixedSleep(acquireMaxRetries, acquireRetryInterval, TimeUnit.SECONDS);
        this.acquireLease(retryPolicy, 0, acquireRetryInterval, 0L, new TracingContext(tracingContext));
        while (this.leaseID == null && this.exception == null) {
            try {
                this.future.get();
            }
            catch (Exception e) {
                LOG.debug("Got exception waiting for acquire lease future. Checking if lease ID or exception have been set", (Throwable)e);
            }
        }
        if (this.exception != null) {
            LOG.error("Failed to acquire lease on {}", (Object)path);
            throw new LeaseException(this.exception);
        }
        LOG.debug("Acquired lease {} on {}", (Object)this.leaseID, (Object)path);
    }

    private void acquireLease(final RetryPolicy retryPolicy, final int numRetries, final int retryInterval, long delay, final TracingContext tracingContext) throws LeaseException {
        LOG.debug("Attempting to acquire lease on {}, retry {}", (Object)this.path, (Object)numRetries);
        if (this.future != null && !this.future.isDone()) {
            throw new LeaseException("There is already an existing lease operation");
        }
        this.future = this.client.schedule(() -> this.client.acquireLease(this.path, -1, tracingContext), delay, TimeUnit.SECONDS);
        this.client.addCallback(this.future, new FutureCallback<AbfsRestOperation>(){

            @Override
            public void onSuccess(@Nullable AbfsRestOperation op) {
                AbfsLease.this.leaseID = op.getResult().getResponseHeader("x-ms-lease-id");
                LOG.debug("Acquired lease {} on {}", (Object)AbfsLease.this.leaseID, (Object)AbfsLease.this.path);
            }

            @Override
            public void onFailure(Throwable throwable) {
                try {
                    if (RetryPolicy.RetryAction.RetryDecision.RETRY == retryPolicy.shouldRetry(null, (int)numRetries, (int)0, (boolean)true).action) {
                        LOG.debug("Failed to acquire lease on {}, retrying: {}", (Object)AbfsLease.this.path, (Object)throwable);
                        AbfsLease.this.acquireRetryCount++;
                        AbfsLease.this.acquireLease(retryPolicy, numRetries + 1, retryInterval, retryInterval, tracingContext);
                    } else {
                        AbfsLease.this.exception = throwable;
                    }
                }
                catch (Exception e) {
                    AbfsLease.this.exception = throwable;
                }
            }
        });
    }

    public void free() {
        if (this.leaseFreed) {
            return;
        }
        try {
            LOG.debug("Freeing lease: path {}, lease id {}", (Object)this.path, (Object)this.leaseID);
            if (this.future != null && !this.future.isDone()) {
                this.future.cancel(true);
            }
            TracingContext tracingContext = new TracingContext(this.tracingContext);
            tracingContext.setOperation(FSOperationType.RELEASE_LEASE);
            this.client.releaseLease(this.path, this.leaseID, tracingContext);
        }
        catch (IOException e) {
            LOG.warn("Exception when trying to release lease {} on {}. Lease will need to be broken: {}", new Object[]{this.leaseID, this.path, e.getMessage()});
        }
        finally {
            this.leaseFreed = true;
            LOG.debug("Freed lease {} on {}", (Object)this.leaseID, (Object)this.path);
        }
    }

    public boolean isFreed() {
        return this.leaseFreed;
    }

    public String getLeaseID() {
        return this.leaseID;
    }

    @VisibleForTesting
    public int getAcquireRetryCount() {
        return this.acquireRetryCount;
    }

    @VisibleForTesting
    public TracingContext getTracingContext() {
        return this.tracingContext;
    }

    public static class LeaseException
    extends AzureBlobFileSystemException {
        public LeaseException(Throwable t) {
            super("Unable to acquire lease: " + t, t);
        }

        public LeaseException(String s2) {
            super(s2);
        }
    }
}

