#
#  event : A Library of Special Functions for Event Histories
#  Copyright (C) 1998 J.K. Lindsey
#
#  This program is free software; you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation; either version 2 of the License, or
#  (at your option) any later version.
#
#  This program is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program; if not, write to the Free Software
#  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
#
#  SYNOPSIS
#
#  pbirth(fr, p, nbin=F, n=20, k=3)
#
#  DESCRIPTION
#
#    Function to fit a generalized negative binomial process as a
# birth process

pbirth <- function(fr, p, nbin=F, n=20, k=3)
{
call <- sys.call()
n <- length(fr)-1
if(missing(p)){
	yi <- seq(0,n)
	ybar <- weighted.mean(yi,fr)
	s2 <- weighted.mean((yi-ybar)^2,fr)
	p <- c(log(-log(ybar/s2)),log(ybar^2/(s2-ybar)))
	if(!nbin)p <- c(p,-1)}
if(nbin) lambda <- function(p,n) exp(p[1])*(exp(p[2])+n)
else lambda <- function(p,n) exp(p[1])*(exp(p[2])+n)^(1-exp(p[3]))
prob <- function(p) {
	x <- matrix(0,n+1,n+1)
	x[matrix(1:(n+1),n+1,2)] <- -lambda(p,0:n)
	i <- matrix(1:n,n,2)
	i[,2] <- i[,2]+1
	x[i] <- lambda(p,0:(n-1))
	pr <- c(1,rep(0,n))%*%mexp(x,n=n,k=k)
	ifelse(pr>0&pr<1,pr,0)}
like <- function(p) -sum(fr*log(prob(p)),na.rm=T)
z0 <- nlm(like,p=p,hessian=T, print.level=0,
	typsiz=rep(1,length(p)), ndigit=10, gradtl=0.00001,
	stepmx=max(sqrt(p%*%p),1),
	steptl=0.0004, itnlim=100, fscale=1)
cov <- solve(z0$hessian)
se <- sqrt(diag(cov))
nn <- sum(fr)
pr <- prob(z0$estimate)
fitted.values <- nn*pr
residuals <- (fr-fitted.values)/sqrt(fitted.values)
z1 <- list(
	call=call,
	intensity=lambda,
	fr=fr,
	maxlike=z0$minimum,
	fitted.values=fitted.values,
	prob=pr,
	residuals=residuals,
	initial.values=p,
	coefficients=z0$estimate,
	se=se,
	cov=cov,
	corr=cov/(se%o%se),
	gradient=z0$gradient,
	error=z0$error,
	code=z0$code)
class(z1) <- "pbirth"
return(z1)}

residuals.pbirth <- function(z) z$residuals
fitted.values.pbirth <- function(z) z$fitted.values
coefficients.pbirth <- function(z) z$coefficients
deviance.pbirth <- function(z) 2*z$maxlike

print.pbirth <- function(z) {
	np <- length(z$coefficients)
	p <- z$initial.values
	nbin <- length(p)==2
	pt <- vector(type="real",np)
	pr <- exp(-exp(z$coefficients[1]))
	nn <- exp(z$coefficients[2])
	pt[1] <- nn/pr-nn
	pt[2] <- nn
	if(!nbin)pt[3] <- 1-exp(z$coefficients[3])
	cat("\nCall:\n",deparse(z$call),"\n\n",sep="")
	t <- deparse(z$intensity)
	cat("Intensity function:",t[2:length(t)],"",sep="\n")
	cat("Initial values:\n")
	cat("Mean  =",p[1],"(",exp(p[2]+exp(p[1]))-exp(p[2]),")\n")
	cat("Shape =",p[2],"(",exp(p[2]),")\n")
	if(!nbin)cat("Power =",p[3],"(",1-exp(p[3]),")\n\n")
	cat("\n-Log likelihood   ",z$maxlike,"\n")
	cat("\nCoefficients:\n")
	coef.table <- cbind(z$coefficients, z$se, pt)
	if(nbin) dn <- c("Mean","Shape")
	else dn <- c("Mean","Shape","Power")
	dimnames(coef.table) <- list(dn,c("estimate", "se","parameter"))
	print.default(coef.table, digits=4, print.gap=2)
	cat("\nCorrelations:\n")
	dimnames(z$corr) <- list(seq(1,np),seq(1,np))
	print.default(z$corr, digits=4)
	invisible(z)}
