#   Copyright 1993, 1994, 1995, 1996  Bank of Canada.
#   Copyright 1997 (June), Paul Gilbert.
#   Copyright 1997 (Aug.), Bank of Canada.
#   Copyright 1998, Bank of Canada.

#   The user of this software has the right to use, reproduce and distribute it.
#   Bank of Canada makes no warranties with respect to the software or its 
#   fitness for any particular purpose. The software is distributed by the Bank
#   of Canada and by Paul Gilbert solely on an "as is" basis. By using the  
#   software, user agrees to accept the entire risk of using this software.

################################################################################



# For installation instructions see the file read.me or the brief user's
#    guide (postscipt file guide.ps).

##############################################################################

############################################################################

#    functions for DSE interface to Time Series Protocol for      <<<<<<<<<<
#      Application Database Interface (TSPADI) data interface    <<<<<<<<<<

############################################################################

# The PADI interface uses some calls to operating system specific functions:
#    -the function sleep is used in TSPADI.function.tests  
#    -the functions whoami and local.host.netname defined in syskern.s
#             (and in the PADI interface software)

############################################################

#   Definition of class c("TSPADIdata", "TSdata") <<<<<<<<<<

############################################################




############################################################

#     methods for TSPADIdata class objects <<<<<<<<<<



############################################################

print.TSPADIdata <- function(data, ...)
  {print.default(data)
   invisible(data)
  }

is.TSPADIdata <-function(obj)
  {r <- "TSPADIdata" == class(obj)[1]
   if(is.na(r)) r <- F
   r
  }


start.TSPADIdata <-function(obj) {obj$start}
end.TSPADIdata <-function(obj) {obj$end}
frequency.TSPADIdata <-function(obj) {obj$freq}
tsp.TSPADIdata <-function(obj)
  {Start <-start(obj)
   End   <-  end(obj)
   f <- frequency(obj)
   if (length(Start)==2) Start <- Start[1] + (Start[2]-1)/f
   if (length(End)==2)   End   <- End[1]   + (End[2]-1)/f
   c(Start, End, f)
  }

periods.TSPADIdata <- function(data) NA  #periods.default(data)
input.periods.TSPADIdata  <- function(data) NA  
output.periods.TSPADIdata <- function(data) NA  

 
input.data.TSPADIdata<-function(data.id, series=seq(input.dimension(data.id)))
{if(is.null(data.id$input)) return(NULL)
 else if(is.null(data.id$input$series)) return(data.id$input[ series ])# form 2
 else return(data.id$input[[ series ]])
}

output.data.TSPADIdata<-function(data.id, series=seq(output.dimension(data.id)))
{if(is.null(data.id$output)) return(NULL)
 else if(is.null(data.id$output$series)) return(data.id$output[series]) #form 2
 else return(data.id$output[[ series ]])  # form 1
}


output.dimension.TSPADIdata <- function(data.id)
  {if(is.null(data.id$output)) return(0)
   else if(is.null(data.id$output$series))return(length(data.id$output))# form 2
   else return(length(data.id$output$series))  # form 1
  }
input.dimension.TSPADIdata <- function(data.id)
  {if(is.null(data.id$input)) return(0)
   else if(is.null(data.id$input$series))return(length(data.id$input))# form 2
   else return(length(data.id$input$series))  # form 1
  }


input.series.names.TSPADIdata <- function(data)
 {# two formats for TSPADIdata are recognized.
 if (is.null(data$input$series))
  {names <- NULL
   for (i in seq(length(data$input)))
      {if (5==length(data$input[[i]])) names<- c(names,data$input[[i]][5])
       else                            names<- c(names,data$input[[i]][2])
      }
  }
 else
  {if (is.null(data$input.names))   names <- data$input$series
   else                             names <- data$input.names
  }
 names
 }

output.series.names.TSPADIdata <- function(data)
 {# two formats for TSPADIdata are recognized.
 if (is.null(data$output$series))
  {names <- NULL
   for (i in seq(length(data$output)))
      {if (5==length(data$output[[i]])) names<- c(names,data$output[[i]][5])
       else                             names<- c(names,data$output[[i]][2])
      }
  }
 else
  {if (is.null(data$output.names))   names <- data$output$series
   else                              names <- data$output.names
  }
 names
 }



identifiers.TSPADIdata <- function(data.id)
  {if (is.null(data.id)) return(NA)
   data.id <- TSdata.name.to.form1(data.id)
   list(input=data.id$input$series,output=data.id$output$series)
  }

source.info.TSPADIdata <- function(data.id)
  {if (is.null(data.id)) return(NA)
   server<- sourceserver(data.id)
   db<- sourcedb(data.id)
   list(input=list(server=server$input, db=db$input),
       output=list(server=server$output, db=db$output))
  }

sourcedb.TSPADIdata <- function(data.id)
  {if (is.null(data.id)) return(NA)
   data.id <- TSdata.name.to.form1(data.id)
   m <-input.dimension(data.id)
   p <-output.dimension(data.id)
   if (m==0) inp<-NULL
   else
     {if (is.null(data.id$input$db))        inp <- rep("",m)
      else if (1==length(data.id$input$db)) inp <- rep(data.id$input$db,m)
      else inp <- data.id$input$db
     }
   if (p==0) out <-NULL
   else
     {if (is.null(data.id$output$db))        out <- rep("",p)
      else if (1==length(data.id$output$db)) out <- rep(data.id$output$db,p)
      else out <- data.id$output$db
     }
   list(input=inp,output=out)
  }


sourceserver.TSPADIdata <- function(data.id)
  {if (is.null(data.id)) return(NA)
   data.id <- TSdata.name.to.form1(data.id)
   m <-input.dimension(data.id)
   p <-output.dimension(data.id)
   if (m==0) inp<-NULL
   else
     {if (is.null(data.id$input$server))     inp <- rep("",m)
      else 
        if (1==length(data.id$input$server)) inp <- rep(data.id$input$server,m)
      else inp <- data.id$input$server
     }
   if (p==0) out <-NULL
   else
     {if (is.null(data.id$output$server))    out<-rep("",p)
      else
       if (1==length(data.id$output$server)) out<-rep(data.id$output$server,p)
      else out <- data.id$output$server
     }
   list(input=inp,output=out)
  }


############################################################

#      Database interface for TSPADIdata  <<<<<<<<<<

############################################################

TSdata.TSPADIdata <- function(data.id)
{ # This function retreives data from a PADI server using getpadi
  # A server specified as NULL or as "" is expanded to the localhost.

  data.id <- TSdata.name.to.form1(data.id)   
  # now assume form 1

 m <-input.dimension(data.id)
 p <-output.dimension(data.id)
 series  <- identifiers(data.id)
 series <- c(series$input, series$output)
 dbname <- sourcedb(data.id)
 dbname <- c(dbname$input, dbname$output)
 server <-sourceserver(data.id)
 server <- c(server$input, server$output)
 server[server==""] <- local.host.netname()

 if (is.null(data.id$spawn.server)) spawn.server<- T
 else spawn.server <- data.id$spawn.server

 if (is.null(data.id$server.process)) server.process<- padi.server.process()
 else server.process <- data.id$server.process

 if (is.null(data.id$cleanup.script)) cleanup.script<- padi.cleanup.script()
 else cleanup.script <- data.id$cleanup.script

 if (is.null(data.id$stop.on.error)) stop.on.error<- T
 else stop.on.error <- data.id$stop.on.error

 if (is.null(data.id$warn)) warn<- T
 else warn <- data.id$warn

 if (is.null(data.id$user)) user <- whoami()
 else user <- data.id$user

 if (is.null(data.id$passwd))  passwd <- ""
 else passwd <- data.id$passwd

 # next 3 lines are to look after older style name forms at the BOC
 ets <- "ets" == substring(dbname,1,3)
 server[ets] <-"ets"
 dbname[ets] <- ""
 
 server[server=="ets"] <- "padi"   # temporary kludge at the BOC
 Start <- start(data.id) 
 if(is.null(Start)) Start <- c(0,0)
 End <- end(data.id) 
 if(is.null(End)) End <- c(0,0)

 tran.in  <- data.id$input.transformations
 tran.out <- data.id$output.transformations
 if (is.null(tran.in))  tran.in  <- rep("", m)
 if (is.null(tran.out)) tran.out <- rep("", p)
 transformations <-  c(tran.in,tran.out)

 if (is.null(data.id$pad)) pad <- F
 else pad <- data.id$pad
 if (is.null(data.id$pad.start)) pad.start <- pad
 else pad.start <- data.id$pad.start
 if (is.null(data.id$pad.end)) pad.end <- pad
 else pad.end <- data.id$pad.end
 data  <- getpadi( series, dbname=dbname, server=server,
                start.server=spawn.server, server.process=server.process,
                cleanup.script=cleanup.script,
                starty=Start[1], startm=Start[2],
                endy=End[1], endm=End[2],
                transformations=transformations,
                pad=(pad|pad.start|pad.end) ,
                user=user, passwd=passwd,
                stop.on.error=stop.on.error, use.tframe=T, warn=warn)
 if (!pad.start) data <- trim.na(data, Start=T, End=F)
 if (!pad.end  ) data <- trim.na(data, Start=F, End=T)
 if (dim(data)[2] != (m+p)) stop("Error retrieving data.")
  
 # retrieving input and output together ensures they are properly aligned.
 if (m==0) input <- NULL
 else 
   input<-tframed(data[,1:m,drop=F], list(start=start(data),
              frequency=frequency(data)), names=input.series.names(data.id))
   
 if (p == 0) output <- NULL  
 else  
   output<-tframed(data[,(m+1):(m+p),drop=F], list(start=start(data),
            frequency=frequency(data)), names=output.series.names(data.id))
 data <- list(input=input, output=output, source=data.id, retrieval.date=date())
 class(data) <- "TSdata"
 data
}


availability.TSPADIdata<-function(data.id, verbose=T)  
{# Indicate  dates for which data is available. 
  data.id <- TSdata.name.to.form1(data.id)   
  # now assume form 1

 m <-input.dimension(data.id)
 p <-output.dimension(data.id)
 series  <- identifiers(data.id)
 series <- c(series$input, series$output)
 dbname <- sourcedb(data.id)
 dbname <- c(dbname$input, dbname$output)
 server <-sourceserver(data.id)
 server <- c(server$input, server$output)
 server[server==""] <- local.host.netname()

 if (is.null(data.id$spawn.server)) spawn.server<- T
 else spawn.server <- data.id$spawn.server

 if (is.null(data.id$server.process)) server.process<- padi.server.process()
 else server.process <- data.id$server.process

 if (is.null(data.id$cleanup.script)) cleanup.script<- padi.cleanup.script()
 else cleanup.script <- data.id$cleanup.script

 if (is.null(data.id$stop.on.error)) stop.on.error<- T
 else stop.on.error <- data.id$stop.on.error

 if (is.null(data.id$warn)) warn<- T
 else warn <- data.id$warn

 if (is.null(data.id$user)) user <- whoami()
 else user <- data.id$user

 if (is.null(data.id$passwd))  passwd <- ""
 else passwd <- data.id$passwd


 if (verbose) 
    names <- c(input.series.names(data.id), output.series.names(data.id))

 # next 3 lines are to look after older style name forms at the BOC
 ets <- "ets" == substring(dbname,1,3)
 server[ets] <-"ets"
 dbname[ets] <- ""
 
 server[server=="ets"] <- "padi"   # temporary kludge at the BOC

 s <- e <- f <- NULL
 for (i in 1:length(series))
      {data <- getpadi(series[i], dbname=dbname[i], server=server[i],
                start.server=spawn.server, server.process=server.process,
                cleanup.script=cleanup.script,
                user=user, passwd=passwd,
                stop.on.error=stop.on.error, use.tframe=T, warn=warn, pad=F)
       s <- c(s, start(data))
       e <- c(e, end(data))
       f <- c(f,frequency(data))
       if (verbose)
         {cat(series[i]," from: ",start(data))
          cat("  to: ",end(data))
          cat("   frequency ", frequency(data))
          if (!is.null(names)) cat("  ",names[i])
          cat("\n")
      }  }
  invisible(list(start=s, end=e, frequency=f))
}

  
putpadi.TSdata   <- function (data, dbname, server=local.host.netname(), 
                   spawn.server=T, server.process=padi.server.process(), 
                   cleanup.script=padi.cleanup.script(),
                   series=series.names(data),
                   user=whoami(), passwd="",
                   stop.on.error=T, warn=T)   
  {#dbname and server can be a single string in which case it is applied to
   # all series. Otherwise it should be a structure like series: a list with
   # elements input and output, each vectors with a string for each series.
   # The returned result (data.id) rearranges this as lists input and output,
   #   each with sublists series, db, and server, which is the form for 
   #   retrieving the data.

   m <-input.dimension(data)
   p <-output.dimension(data)

   if(!is.list(dbname)) 
     {z <-dbname[1]
      dbname <- list()
      if (m!=0) dbname$input  <-rep(z,m)
      if (p!=0) dbname$output <-rep(z,p)
     }

   if(!is.list(server)) 
     {z <-server[1]
      server <- list()
      if (m!=0) server$input  <-rep(z,m)
      if (p!=0) server$output <-rep(z,p)
     }
   if (m!=0) server$input [server$input==""]  <- local.host.netname()
   if (p!=0) server$output[server$output==""] <- local.host.netname()

   # next 3 lines are to look after older style name forms at the BOC
   if (m!=0)
     {ets <- "ets" == substring(dbname$input,1,3)
      server$input[ets] <-"ets"
      dbname$input[ets] <- ""
     }
   if (p!=0)
     {ets <- "ets" == substring(dbname$output,1,3)
      server$output[ets] <-"ets"
      dbname$output[ets] <- ""
     }
   server$input[server$input=="ets"]  <- "padi" #temporary kludge at the BOC
   server$output[server$output=="ets"]<- "padi" #temporary kludge at the BOC

   data.id <- list(input= list(series=series$input,  db=dbname$input, 
                              server=server$input),
                   output=list(series=series$output, db=dbname$output, 
                              server=server$output),
                   spawn.server=spawn.server, 
                   server.process=server.process,
                   cleanup.script=cleanup.script,
                   user=user, passwd=passwd,
                   stop.on.error=stop.on.error, warn=warn)
   if (!is.null(input.data(data)))
     {if(all (1 == start(input.data(data))))
         warning("Fame may choke on a start date of 1,1")
      mat <- ts(input.data(data), start = start(input.data(data)), 
                frequency=frequency(input.data(data)))

      ok <-putpadi(mat ,  dbname=dbname$input, series.names=series$input,
                   server=server$input,
                   start.server=spawn.server, server.process=server.process,
                   cleanup.script=cleanup.script,
                   user=user, passwd=passwd,
                   stop.on.error=stop.on.error, warn=warn)
     }
   if (!is.null(output.data(data))) 
     {if(all (1 == start(output.data(data))))
         warning("Fame may choke on a start date of 1,1")
      mat <- ts(output.data(data), start = start(output.data(data)),
                 frequency=frequency(data))
      ok <-putpadi(mat ,  dbname=dbname$output, series.names=series$output,
                   server=server$output,
                   start.server=spawn.server, server.process=server.process,
                   cleanup.script=cleanup.script,
                   user=user, passwd=passwd,
                   stop.on.error=stop.on.error, warn=warn)

     }
   data.id$pad <- F
   class(data.id) <- c("TSPADIdata", "TSdata")
   invisible(data.id)
  }


#   The following function is supplied separately (with PADI ). The 
#   documentation is included here so it will integrate with DSE.


set.data <- function()
 {# prompt for input and output series identifiers, set class, etc.
  cat("This function prompts for the names and database locations for\n")
  cat("input and output series, until an empty line is entered.\n")
  cat("If your model has no input (exogenous variable) then return an empty line.\n\n")
  input.series <- NULL
  input.db <- NULL
  input.transformations <- NULL
  output.series <- NULL
  output.db <- NULL
  output.transformations <- NULL
  cat("Input (exogenous) variables...\n")
  repeat
    {cat("  series..");key <- readline()
     if (""== key) break
     else input.series <-c(input.series,key)
     cat("  database..");key <- readline()
     input.db <-c(input.db,key)
     cat("  transformation..");key <- readline()
     if (""== key) key <- "c"
     input.transformations <-c(input.transformations,key)
    }  
  cat("Output (endogenous) variables...\n")
  repeat
    {cat("  series..");key <- readline()
     if (""== key) break
     else output.series <-c(output.series,key)
     cat("  database..");key <- readline()
     output.db <-c(output.db,key)
     cat("  transformation..");key <- readline()
     if (""== key) key <- "c"
     output.transformations <-c(output.transformations,key)
    }
  if (is.null(input.series))  
     data <- list( output=list(series=output.series,db=output.db), 
                   output.transformations=output.transformations)
  else
     data <- list(input=list(series=input.series,db=input.db),
                  input.transformations=input.transformations,
                  output=list(series=output.series,db=output.db),
                  output.transformations=output.transformations)
  class(data) <-c("TSPADIdata","TSdata")
  cat("  starting year..");key <- readline()
     if (!(""== key)) 
       {start <- as.integer(key)
        cat("  starting period..");key <- readline()
        start <- c(start, as.integer(key))
        if(any(is.na(start))) cat("Warning: start improperly specified. NOT set!")
        else data$start <- start
        }
  cat("  ending year..");key <- readline()
     if (!(""== key)) 
       {end <- as.integer(key)
        cat("  ending period..");key <- readline()
        end <- c(end, as.integer(key))
        if(any(is.na(end))) cat("Warning: end improperly specified. NOT set!")
        else data$end <- end
        }

  cat("The series may now be retrieved, in which case the data is\n")
  cat("  fixed as currently available, or they may be left `dynamic',\n")
  cat("  in which case they are retrieved whenever the result of this\n")
  cat("  function (set.data) is referenced.\n")
  cat("Retrieve data y/n:");key <- readline()
  if ((key =="y") | (key=="Y")) data <- TSdata(data)
  data
}


#######################################################################

#     functions for converting defunct format FAMEdata structure
#         (these are primarily for use at the BOC)

#######################################################################

TSdata.FAMEdata <-function(data)
  {stop("FAMEdata is defunct. Use FAMEdata.to.TSPADIdata to convert the structure")}

FAMEdata.to.TSPADIdata <-function(data)
  {# TSdata.name.to.form1(data) no longer works with FAMEdata objects so
   # form2 is handled in the first if and form 1 in the else
   if(is.null(data$output$series) & is.null(data$input$series))
      {if(!is.null(data$input))
         {input <- list()
          for(i in seq(length(data$input)))
             {inp <-c("",(data$input)[[i]]) 
              if ("ets" == substring(inp[2],1,3) ) inp[1:2] <- c("ets","")
              input <- append(input, list(inp))
             }
          data$input <- input
         }
       if(!is.null(data$output))
         {output <- list()
          for(i in seq(length(data$output)))
             {out <-c("",(data$output)[[i]]) 
              if ("ets" == substring(out[2],1,3) ) out[1:2] <- c("ets","")
              output <- append(output, list(out))
             }
          data$output <- output
         }

      }
   else
      {stop("form 1 names not done yet.")
      }
   class(data) <- c("TSPADIdata","TSdata")
   data
  }

#######################################################################


#######################################################################

#    TS PADI interface tests (from Brief User's Guide)   <<<<<<<<<<

#######################################################################

TSPADI.function.tests <- function( verbose=T, synopsis=T,
      fuzz.small=1e-14, fuzz.large=1e-6, ets=F)
{# test for TSPADI access using simple.server
 # and if ets=T then run example from Brief User's guide (requires ets database)

 # These tests only check that the DSE structures work with PADI. For a more
 #   complete set of PADI tests see the file padi.s distributed 
 #   with the TS PADI software.


  if (synopsis & !verbose) cat("DSE TSPADI tests ...")

  scratch.db <-"zot123456.db"
  unlink(scratch.db)
  server <- local.host.netname()

 if (verbose) cat("DSE TSPADI test 0 ... ")
  if (check.padi.server(server))
     stop("A server is already running. Testing stopped. Use cleanup.padi.server() or kill.padi.server() to terminate it.")

  pid <- start.padi.server(server=server, dbname="", 
                 server.process=paste("simple.server ", scratch.db))
  on.exit(cleanup.padi.server(pid, cleanup.script="cleanup.simple.server"))

  # wait to ensure padi server is started
     for (i in 1:30)
       {if (check.padi.server(server)) break
        sleep(1)
       }

  exp1 <- tframed(matrix(1*exp(1:20),20,1), list(start=c(1950,1),freq=1))
#  exp1 <- ts(1*exp(1:20), start=c(1950,1),freq=1)
#  tframe(exp1) <- tframe(exp1)
  eg.put.data <- list(input= exp1, 
                      input.names="exp1",
                      output= tframed(tbind(2*exp1, 3*exp1),tframe(exp1)), 
                      output.names=c("exp2","exp3"))
#  exp1 <- ts(1*exp(1:20), start=c(1950,1),freq=1)
#  eg.put.data <- list(input= tsmatrix(exp1), 
#                      input.names="exp1",
#                      output= tsmatrix(2*exp1, 3*exp1), 
#                      output.names=c("exp2","exp3"))
  class(eg.put.data) <- c("TSdata")
  eg.names <- putpadi.TSdata(eg.put.data,
                      dbname=scratch.db, server=server,
                      spawn.server=T, server.process="simple.server", 
                      cleanup.script="cleanup.simple.server",
                      stop.on.error=T, warn=T )
  ok<-is.TSPADIdata(eg.names) 
  all.ok <- ok
  if (verbose) 
    {if (ok) cat("ok\n")
     else  cat("failed! putpadi server started\n")
    }

  if (verbose) cat("DSE TSPADI test 1 ... ")
  eg.data <- TSdata(eg.names)
  ok <- is.TSdata(eg.data ) & test.equal(eg.data, eg.put.data, fuzz=fuzz.large)
  all.ok <- all.ok & ok 
  if (verbose) 
    {if (ok) cat("ok\n")
     else    cat("failed!\n")
    }

  if (verbose) cat("DSE TSPADI test 2 ... ")
  eg.names <- list(
	input=list(series=c( "exp1","exp2"), server=NULL, db=scratch.db), 
 	input.names=c( "exp1","exp2"),
	output=list(series=c( "exp1","exp2","exp3"), server=NULL, db=scratch.db), 
 	output.names=c( "exp1","exp2","exp3") ,
        spawn.server=T, server.process="simple.server", 
        cleanup.script="cleanup.simple.server", stop.on.error=T, warn=T )

  class(eg.names) <- c("TSPADIdata", "TSdata")
  eg.data <- TSdata(eg.names)
  ok <- is.TSdata(eg.data ) 
    (max(abs(output.data(eg.data) - 
              cbind(exp(1:20),2*exp(1:20),3*exp(1:20)) ))<fuzz.large)
  all.ok <- all.ok & ok 
  if (verbose) 
    {if (ok) cat("ok\n")
     else    cat("failed!\n")
    }

  if (verbose) cat("DSE TSPADI test 3 ... ")
  #  form 2 data names
  eg.names <- list(
	input= list(c(server,scratch.db,"exp1", "", "exp1")),
	output=list(c(server,scratch.db,"exp1", "", "exp1"),
                   c(server,scratch.db,"exp2", "", "exp2"),
                   c(server,scratch.db,"exp3", "", "exp3"))   ,
        spawn.server=T, server.process="simple.server", 
        cleanup.script="cleanup.simple.server", stop.on.error=T, warn=T )

  class(eg.names) <- c("TSPADIdata", "TSdata")
  eg.data <- TSdata(eg.names)
  ok <- is.TSdata(eg.data)  & 
    (max(abs(output.data(eg.data) - 
              cbind(exp(1:20),2*exp(1:20),3*exp(1:20)) ))<fuzz.large)
  all.ok <- all.ok & ok 
  if (verbose) 
    {if (ok) cat("ok\n")
     else    cat("failed!\n")
    }

  if (verbose) cat("DSE TSPADI test 4 ... ")
  avail <- availability(eg.names, verbose=F)
  ok <- all(c(avail$start ==  rep(c(1950,1),4),
              avail$end   ==  rep(c(1969,1),4),
              avail$frequency ==  rep(1,4)))

  all.ok <- all.ok & ok 
  if (verbose) 
    {if (ok) cat("ok\n")
     else    cat("failed!\n")
    }

  on.exit()
  cleanup.padi.server(pid, cleanup.script="cleanup.simple.server")

  if (synopsis) 
    {if (verbose) cat("All DSE TSPADI tests completed")
     if (all.ok) cat(" ok\n")
     else    cat(", some failed!\n")
    }

if (ets)
{# test examples for TSPADI access (from Brief User's guide)
   # wait to ensure padi server is terminated
     for (i in 1:30)
       {if (!check.padi.server(server)) break
        sleep(1)
       }
   
  if (synopsis & !verbose) cat("DSE TSPADI/ets tests ...")

  if (verbose) cat("DSE TSPADI/ets test 1 ... ")
  eg2.DSE.data.names <- list(
	output=list(series=c( "I37005"), server=c("ets"), db=NULL), #etsgdpfc
 	output.names=c( "manuf.prod.") ,
        spawn.server=T, server.process="fame.server", 
        cleanup.script="cleanup.fame.server", stop.on.error=T, warn=T )

  class(eg2.DSE.data.names) <- c("TSPADIdata", "TSdata")
  eg2.DSE.data <- TSdata(eg2.DSE.data.names)
  ok <- is.TSdata(eg2.DSE.data )
  all.ok <- ok 
  if (verbose) 
    {if (ok) cat("ok\n")
     else    cat("failed!\n")
    }

  if (verbose) cat("DSE TSPADI/ets test 2 ... ")
  eg3.DSE.data.names <- list(
 	input=list(series="lfsa455", server="ets", db= NULL), #etslabour
 	input.transformations= "percent.change.vector",
 	input.names= "manuf.emp.",
 	output=list(series=c( "i37005"), server="ets", db= NULL), #etsgdpfc 
 	output.names=c( "manuf.prod."),
 	output.transformations= c("percent.change.vector"),
 	pad.start=F,
 	pad.end =T,
        spawn.server=T, server.process="fame.server", 
        cleanup.script="cleanup.fame.server", stop.on.error=T, warn=T )

  class(eg3.DSE.data.names) <- c("TSPADIdata", "TSdata")
  eg3.DSE.data <- TSdata(eg3.DSE.data.names)
  ok <- is.TSdata(eg3.DSE.data )
  all.ok <- all.ok & ok 
  if (verbose) 
    {if (ok) cat("ok\n")
     else    cat("failed!\n")
    }

  if (verbose) cat("DSE TSPADI/ets test 3 ... ")
  JofF.VAR.data.names <- list(
	input = list(series= c("B14017"), server="ets", db= NULL), #etsmfacansim
	input.transformations= c("diff"),
	input.names=c("R90"),
	output = list(series=c("P484549", "I37026", "b1627", "b14013",
				 "b4237", "D767608", "b3400", "M.BCPI",
				 "M.JQIND", "M.CUSA0"), server="ets", db= NULL),
                 # etscpi etsgdpfc etsmfacansim etsmfacansim etsdistscu
                 # etslabour etsforexch etsbcpi etsusa etsusa
  	output.transformations=c("percent.change.vector", 
			"percent.change.vector","percent.change.vector",
			"diff", "diff", "percent.change.vector",
			"percent.change.vector", "percent.change.vector",
			"percent.change.vector", "percent.change.vector"),
	output.names=c("CPI", "GDP", "M1", "RL", "TSE300", 
			"employment", "PFX", "com. price ind.", 
			"US ind. prod.", "US CPI"),
        spawn.server=T, server.process="fame.server", 
        cleanup.script="cleanup.fame.server", stop.on.error=T, warn=T )
  class(JofF.VAR.data.names) <- c("TSPADIdata", "TSdata")
  JofF.VAR.data <- TSdata(JofF.VAR.data.names)
  ok <- is.TSdata(JofF.VAR.data )
  all.ok <- all.ok & ok 
  if (verbose) 
    {if (ok) cat("ok\n")
     else    cat("failed!\n")
    }

  if (verbose) cat("DSE TSPADI/ets test 4... ")
  alt.JofF.VAR.data.names <- list(
	input = list(c("ets","","B14017","diff","R90")),
	output = list(
         c("ets","", "P484549","percent.change.vector", "CPI"),
         c("ets","", "I37026", "percent.change.vector", "GDP"),
         c("ets","", "b1627",  "percent.change.vector", "M1"),
         c("ets","", "b14013",  "diff", "RL"),
         c("ets","", "b4237",   "diff", "TSE300"),
         c("ets","", "D767608",  "percent.change.vector", "employment"),
         c("ets","", "b3400",  "percent.change.vector", "PFX"),
         c("ets","", "M.BCPI",  "percent.change.vector", "com. price ind."),
         c("ets","", "M.JQIND",  "percent.change.vector", "US ind. prod."),
         c("ets","", "M.CUSA0",  "percent.change.vector", "US CPI")),
        spawn.server=T, server.process="fame.server", 
        cleanup.script="cleanup.fame.server", stop.on.error=T, warn=T )

  class(alt.JofF.VAR.data.names) <- c("TSPADIdata", "TSdata")
  alt.JofF.VAR.data <- TSdata(alt.JofF.VAR.data.names)
  ok<- is.TSdata(alt.JofF.VAR.data) &
                 test.equal(JofF.VAR.data, alt.JofF.VAR.data)
  all.ok <- all.ok & ok 
  if (verbose) 
    {if (ok) cat("ok\n")
     else    cat("failed!\n")
    }

  if (synopsis) 
    {if (verbose) cat("All DSE TSPADI/ets tests completed")
     if (all.ok) cat(" ok\n")
     else    cat(", some failed!\n")
    }
}
invisible(all.ok)
}
#   Copyright 1993, 1994, 1995, 1996  Bank of Canada.
#   Copyright 1997 (June), Paul Gilbert.
#   Copyright 1997 (Aug.), Bank of Canada.
#   Copyright 1998, Bank of Canada.

#   The user of this software has the right to use, reproduce and distribute it.
#   Bank of Canada makes no warranties with respect to the software or its 
#   fitness for any particular purpose. The software is distributed by the Bank
#   of Canada and by Paul Gilbert solely on an "as is" basis. By using the  
#   software, user agrees to accept the entire risk of using this software.

################################################################################



# This file contains functions which do not work correctly or have not been completely tested.

eigen.function <- function(mod1,mod2,alpha)
{mod1$parms <- mod1$parms+alpha*(mod2$parms-mod1$parms)
 roots(set.arrays(mod1))
}

eigen.trace <- function(mod1,mod2, inc=10)
{#plot the trace of the eigenvalues on a straight line (in parameter space)
 # from mod1 to mod2.
 # inc gives the number of increments
 eigen.plot(mod1,pch='+')
 m <-mod1
 for (alpha in 1:(inc-1))
   {m$parms <- mod1$parms+alpha*(mod2$parms-mod1$parms)/inc
    add.eigen.plot(set.arrays(m), pch='-')
   }
 add.eigen.plot(mod2,pch='>')
 invisible()
}

eval.trace <- function(mod1,mod2, alpha, replications=100, seed=NULL, 
                       simulation.args=NULL,
                       estimation="black.box", estimation.args=list(verbose=F), 
                       criterion ="parms", criterion.args =NULL)
{#for models on a straight line (in parameter space) from mod1 to mod2
 # evaluate an estimation routine.
 # inc gives the number of increments
 m <-mod1
 m$parms <- mod1$parms+alpha*(mod2$parms-mod1$parms)
# for (alpha in 1:(inc-1))
#   {m$parms <- mod1$parms+alpha*(mod2$parms-mod1$parms)/inc
#    add.eigen.plot(set.arrays(m), pch='-')
#   }
 eval.estimation( set.arrays(m), replications=replications, seed=seed, 
                       simulation.args=simulation.args,
                       estimation=estimation, estimation.args=estimation.args, 
                       criterion =criterion, criterion.args =criterion.args)
}

trace.model.roots<- function(m1,m2, steps=100)
{# trace the roots of models from m1 to m2 on a straight line in parameter space
   m <- m1
   s <- roots(m1)
   stab <- matrix(NA,1+steps, length(s))
   stab[1,] <- s
   for (i in 1:steps)
     {cat(i," ")
      m$parms <- m1$parms+ (m2$parms-m1$parms)*i/steps
      stab[1+i,] <- roots(set.arrays(m))
     }
   plot(x=seq(nrow(stab)), y=cbind(Re(stab), Im(stab)))
   invisible(stab)
}

trace.roots.error<- function(obj)
{# graph of roots for straight line from last cum parameter values to truth
   if ((!is.TSmodel(obj)) & (!is.TSestModel(obj))) TS.error.exit()
   N<-length(obj$result)
   if (is.TSestModel(obj$model)) m1 <- obj$model$model
   else m1<- obj$model
   r <- matrix(0,N,length(m1$parms))
   if (is.TSestModel(obj)) for (i in 1:N) r[i,] <- obj$result[[i]]$model$parms
   else    for (i in 1:N) r[i,] <- obj$result[[i]]$parms
   r[is.inf(r)] <- 0
   m2 <-m1
   m2$parms <- apply(r,2,sum)/N
   invisible(trace.model.roots(m1, set.arrays(m2)))
}



em <- function(model,  max.iter=10, ftol=1e-5) 
{# model estimation with EM algorithm       - NOT WORKING
 # model is an object of class TSestModel (with initial parameter values and data). 
 # max.iter is an integer indicating the maximum number of iterations.
 # The value returned is an object of class TSestModel with an additional
 #  element, $converged, which is T or F, indicating convergence.

warning("This procedure has not been debugged or tested!")

 if (!is.TSestModel(model)) 
    {cat("This function requires an arguement of class TSestModel.\n")
     return("This function requires an arguement of class TSestModel.\n")
    }
 data <- TSdata(model$data)
 est  <- model$estimates
 model <- model$model
 if  (!is.noninnov.SS(model))
    {cat("This function requires a model of class SS  non-innov.")
     return("This function requires a model of class SS  non-innov.")
    } 
 iter <- 0 
 converged <-F 
 cat("   iteration  0")
 smodel <- smoother(model,data)
 tL <-smodel$estimates$like[1]  #+ L(model$smooth$state-model$estmates$state)
 cat(": " , tL,"\n")
 while ((iter <= max.iter) & ( !converged) ) { 
   iter     <-  iter + 1
   cat("   iteration " , iter)
   model<- calc.parms(smodel)
   smodel <- smoother(model,data)
   if (abs(tL - smodel$estimates$like[1]) <= ftol) converged <-T
   tL <-smodel$estimates$like[1]
   cat(": " , tL,"\n")
 } 
smodel$converged <- converged
smodel
} 
   
 calc.parms <- function(smodel)
  {#regression to get new model based on conditional (smoothed) state.
   # see Shumway & Stoffer
# this fails if the state dimension is 1. It could be easily fixed if drop= were allowed
#   to be a vector. Unfortunately, the code assumes the time subscript will be dropped
# but not the other dimensions. (a fortran version may work.)

warning("This procedure has not been debugged or tested!")

   smooth<-smodel$smooth         # smooth$track is smoother tracking error P(t|N) .. $state
   filter <-smodel$filter
   data <- smodel$data
   Pt.t1 <- smodel$filter$track   #filter track error P(t|t-1)
   model <-smodel$model
   H <- model$H
   FF <-model$F
   QQ <- model$Q %*% t(model$Q)
   RR <- model$R %*% t(model$R)
   if (is.null(model$z0)) z0 <- rep(0,nrow(FF))
   else z0 <- model$z0
   if (is.null(model$P0)) P0 <- diag(1,nrow(FF))
   else P0 <- model$P0
   sampleT <- dim(smooth$state)[1]
   Pt.t <- array(0,dim(Pt.t1)) # for P(t|t) 
   for (Time in 1:sampleT) 
      {K <- filter$track[Time,,]%*%t(H)%*% 
              solve(H%*% filter$track[Time,,] %*% t(H) + RR) #(A5)
       P <- filter$track[Time,,] - K %*% H %*% filter$track[Time,,]  #P(t|t) (A7)
       Pt.t[Time,,] <- (P+t(P))/2 #force symmetry to avoid rounding problems
      }
   J <- array(NA,dim(Pt.t))  #Shumway & Stoffer (A8)
   for (Time in 1:(sampleT-1)) 
      J[Time,,]  <-Pt.t[Time,,] %*% t(FF) %*% solve(Pt.t1[Time+1,,])
   Ptt1.n <- array(0,dim(Pt.t))  #Shumway & Stoffer (A11) #cov(z(t), z(t-1)| N)
   # K still contains gain at sampleT
   Ptt1.n[sampleT,,] <- (diag(1,dim(FF)[1]) - K %*% H)%*% FF %*% Pt.t[sampleT-1,,]
#   for (Time in (sampleT-1):2)
#      Ptt1.n[Time,,]  <- Pt.t[Time,,]%*% t(J[Time-1,,]) + J[Time,,] %*%
#         (Ptt1.n[Time+1,,] -  FF%*%Pt.t[Time,,])%*%t(J[Time-1,,])
#   Ptt1.n[1,,] <-Pt.t[1,,]     # this is not exact
   for (Time in 1:(sampleT-1))  # Levy & Porter Appendix B (B 11)
      Ptt1.n[Time,,]  <- J[Time,,] %*% smooth$track[Time+1,,]

   C <- apply(smooth$track,c(2,3),sum) + t(smooth$state) %*% smooth$state
    # this may not be correct
   A <- C+P0+outer(z0,z0) -smooth$track[sampleT,,] - outer(smooth$state[sampleT,], smooth$state[sampleT,])
   #B <- apply(P,c(2,3),sum)                        # this is not correct
   #B <- B + t(smooth$state[2:sampleT,]) %*% smooth$state[1:(sampleT-1),]
   B <- apply(Ptt1.n,c(2,3),sum) 
   B <- B + t(smooth$state[2:sampleT,]) %*% smooth$state[1:(sampleT-1),]
#browser()
   v <- svd(A)
#   if(1 == length(v$d)) A.inv <- v$v %*% (1/v$d) %*% t(v$u)
#   else A.inv <- v$v %*% diag(1/v$d) %*% t(v$u) # more robust than solve(A)
#  following is equivalent
   A.inv <-  v$v %*% (t(v$u) * 1/v$d) 
   model$F <- B %*%  A.inv
   QQ <- (C - B%*%A.inv%*%t(B))/sampleT
   model$Q <- t(chol((QQ+t(QQ))/2))  # insure symmetry (avoid rounding problems)
   #sm.pred <- matrix(NA,sampleT,dim(model$H)[1])
   #for (Time in 1:sampleT) sm.pred[t,] <- H %*% smooth$state[Time,]
   sm.pred <- t( H %*% t(smooth$state))
   e <- output.data(data)- sm.pred
   e <- t(e) %*% e    # cov matrix of errors
   MPM <- diag(0,dim(H)[1])
   for (Time in 1:sampleT) MPM <- MPM + H %*% smooth$track[Time,,] %*% t(H) 
   RR <- (e+MPM)/sampleT
   model$R <- t(chol((RR+t(RR))/2))  # insure symmetry (avoid rounding problems)
#browser()
   set.parameters(model) 
  }



est.SS.Aoki  <- function(data, n=NULL)
{#estimate autocov then get nested-balanced state space model by svd a la Aoki (more or less).
warning("Aoki's procedure is theoretically flawed. There is an error ",
  "in Aoki's main theorem. To go from (4) to (5) in Section 6, Appendix ",
  "A.6, p267 (State Space Modeling of Time Series, Second, Revised and ",
  "Enlarged Edition) requires that conj(lamda)*lamda = 1, but also ",
  "conj(lamda)-lamda = 0, which means not only that the modulus must be ",
  "1, but also that lamda must be 1. I am not aware of the specific ",
  "error being documented elsewhere, but the incorrectness of the result ",
  "has been pointed out in Heij, Kloek and Lucas Econometric Reviews ",
  "11:379-396 (1992) and by Vaccaro and Vukina J.of Economic Dynamics ",
  "and Control 17:401-421 (1993). The technique breaks down except in ",
  "some special data situations. The error can be seen by following ",
  "through this code using browser statements. Aoki has suggested some ",
  "corrections, but the correction proposed by Vaccaro and Vukina seems ",
  "preferable.  That technique has been partially implemented in this ",
  "library (but is not fully debugged and tested and may be omitted). It ",
  "is computationally fairly intensive, and also requires considerable ",
  "user intervention. Users might instead consider the Mittnik reduction ",
  "as used in the bft procedure.")
  data <- TSdata(data)
  model <-reduction.Aoki.from.Hankel( acf.M.TSdata(data), data=data, n=n )
  model$description <- "nested-balanced model from covariance estimates a la Aoki (more or less)"
  KF(model, data)
}

est.SS.Akaike.v0<- function # estimate state space model using Akaike's canonical correlation - not working
   (data, max.lag=6, Goodrich=T)
{ #  estimate a state space model by canonical correlation of the future against the past.
# If Goodrich=T then present is in Past but not Future
#    otherwise  present is in both Past and Future.
  # see Akaike ...   but so far this procedure does not follow him.
  # max.lag determines the maximum shift for future and past.

warning("This procedure has not been debugged or tested!")

 
  data <- TSdata(data)
  m <- input.dimension(data)
  if(is.null(m))  m <- 0
  p <- output.dimension(data)
  N <- periods(data)
  if ( (m != 0) && (N != input.periods(data)))
     stop("sample size must be equal for input and output series.")
 # NA would be better than truncation when shifting but cancor does not yet accept them.
      # shift input to give feedthrough in one period
  if (m != 0)
    {z <- cbind(input.data(data)[2:N,],output.data(data)[1:(N-1),])
     N<-N-1
     output.data(data) <- output.data(data)[1:N,]
    }
  else z <- output.data(data)
  Past <- matrix(NA,N-max.lag,(p+m)*(1+max.lag))
  for (i in 0:max.lag) 
    Past[,(1+(m+p)*i):((m+p)*(1+i))] <-z[(max.lag-i+1):(N-i),]
  if (Goodrich)
    {Future <- matrix(NA,N-max.lag,p*max.lag)
     for (i in 1:max.lag)
     Future[,(1+p*(i-1)):(p*i)] <-output.data(data)[(1+i):(N+i-max.lag),]
    }
  else
    {Future <- matrix(NA,N-max.lag,p*(1+max.lag))
     for (i in 0:max.lag)
     Future[,(1+p*i):(p*(1+i))] <-output.data(data)[(1+i):(N+i-max.lag),]
    }
  CCor <- cancor(Past,Future)
  n <- 1         # for some reason this seems to work best ??!!
#  cat("Canonical correlations between future and past:\n")
#  print(CCor$cor)
#  n <-eval(parse(prompt="Enter the state dimension (enter 0 to stop): "))
#  if( n<1) stop("TERMINATED! DIMENSION MUST BE GREATER THAN 0!")
#  the following deviates from Akaike...??   also see version below
  if (n==1) M <-outer(CCor$ycoef[1:p,1], CCor$xcoef[,1])
  else      M <-CCor$ycoef[1:p,1:n] %*% diag(CCor$cor) %*% t(CCor$xcoef[,1:n])# check!
  browser()
  model <-balance.Mittnik.svd(M,data=data)$model
  model$description <-
    "model estimated by canonical correlation of future against past a la Akaike and nested-balanced a la Mittnik"
  KF(model, data)
}
est.SS.Akaike  <- function # estimate state space model using Akaike's canonical correlation - not working
   (data, max.lag=6, Goodrich=T)
{ #  estimate a state space model by canonical correlation of the future against the past.
  # see Akaike ...   but so far this procedure does not follow him.
  # max.lag determines the maximum shift for future and past.
 
  data <- TSdata(data)
  m <- ncol(input.data(data))
  if(is.null(m))  m <- 0
  p <- ncol(output.data(data))
  N <- nrow(output.data(data))
  if ( (m != 0) && (N != nrow(input.data(data)))) return("sample size must be equal for input and output series.")
 # NA would be better than truncation when shifting but cancor does not yet accept them.
      # shift input to give feedthrough in one period
  if (m != 0)
    {z <- cbind(input.data(data)[2:N,],output.data(data)[1:(N-1),])
     N<-N-1
     output.data(data) <- output.data(data)[1:N,]
    }
  else z <- output.data(data)
  Past <- matrix(NA,N-2*max.lag,(p+m)*(1+max.lag))
  for (i in 0:max.lag) 
    Past[,(1+(m+p)*i):((m+p)*(1+i))] <-z[(max.lag+1-i):(N-max.lag-i),]
  if (Goodrich)
    {Future <- matrix(NA,N-2*max.lag,p*max.lag)
     for (i in 1:max.lag)
     Future[,(1+p*(i-1)):(p*i)] <-output.data(data)[(max.lag+1+i):(N-max.lag+i),]
    }
  else
    {Future <- matrix(NA,N-2*max.lag,p*(1+max.lag))
     for (i in 0:max.lag)
     Future[,(1+p*i):(p*(1+i))] <-output.data(data)[(max.lag+1+i):(N-max.lag+i),]
    }
  CCor <- cancor(Past,Future)
  n <- 1         # for some reason this seems to work best ??!!
#  cat("Canonical correlations between future and past:\n")
#  print(CCor$cor)
#  n <-eval(parse(prompt="Enter the state dimension (enter 0 to stop): "))
#  if( n<1) stop("TERMINATED! DIMENSION MUST BE GREATER THAN 0!")
#  the following deviates from Akaike...??
  if (n==1) M <-outer(CCor$ycoef[1,1:p], CCor$xcoef[1,])
#       M <-outer(CCor$ycoef[1:p,1], CCor$xcoef[,1]) this doesn't seem to work as well.
  else      M <-CCor$ycoef[1:p,1:n] %*% diag(CCor$cor[1:n]) %*% t(CCor$xcoef[,1:n])
# check! Above seems to work better for n!=1 but is not consistent.
# In any case, n=1 seems to work best!! for some reason?
# or M <-t(CCor$ycoef[1:n,1:p]) %*% diag(CCor$cor[1:n]) %*% CCor$xcoef[1:n,]
  browser()
  model <-balance.Mittnik.svd(M,data=data)$model
  model$description <-
    "model estimated by canonical correlation of future against past a la Akaike and nested-balanced a la Mittnik"
    KF(model, data)
}

reduction.Aoki <- function # nested-balanced state space model reduction by svd of Hankel generated from a model
   (model, n=NULL, data=NULL){
 # if state dim. n is supplied then criteria are not calculated and given n is used.
 # data is only necessary if selection criteria (AIC, etc.) are to be calculated

warning("This procedure has not been debugged or tested!")

  if (is.TSestModel(model)) model <- model$model
  if (!is.TSmodel(model)) TS.error.exit()
  if      (is.SS(model))   m <-ncol(model$G)
  else if (is.ARMA(model)) m <-dim(model$C)[3]
  if(is.null(m)) m<-0
  reduction.Aoki.from.Hankel(acf.M.TSmodel(model), m=m, n=n, data=data)
}

reduction.Aoki.from.Hankel<- function # calculate a nested-balanced state space model by svd
   (M, m=NULL, n=NULL, data=NULL)
{ # Calculate a nested-balanced state space model by svd a la Aoki.
  # The program calculates models and several criteria for all state dimensions 
  # up to the order specified(prompted for). This can be skipped by specifing 0.
  # The returned model is of the dimension (n) specified.
  # If state dim. n is supplied then criteria are not calculated and given n is used.
  # If data is supplied and n is not supplied then criteria tests are 
  #   calculated and there is a prompted for n.
  # For the Aoki approach M is autocovariances including u, 
  #     that is, the first row of the Hankel matrix.
  # m is the dimension of input series. It is necessary only if data is not supplied.
  # The output dimension p is taken from nrow(M).
  # data is necessary only if criteria (AIC,etc) are to be calculated.
  # REFERENCES
  #
  #   - Aoki and Havenner(1991),State Space Modeling of Multiple Time Series",
  #       Econometric Reviews, v10,No.1.

warning("This procedure has not been debugged or tested!")

  
set.model.Aoki<-function(m,n,p, c.mtr,c.inv,o.mtr,o.inv,shifted.Hkl,Gamma0) #local
{ H  <- o.mtr[1:p,1:n,drop=F]           # this truncation is not quite (3.30)
  FF <- o.inv[1:n,,drop=F] %*% shifted.Hkl%*%c.inv[,1:n,drop=F]
  if (m == 0) G <- NULL             # no exog.
  else        G <- c.mtr[1:n,1:m,drop=F]  # exog.
  Om <- c.mtr[1:n,(m+1):(m+p),drop=F]   # this truncation is not quite (3.31)
  Gamma0.inv <- solve(Gamma0)
  D <- t(FF) - t(H) %*% Gamma0.inv %*% t(Om)   
  D.inv <- solve(D)
  CGCD <- t(H)%*% Gamma0.inv %*% H %*%t(D.inv)
  OmGOm <- Om %*% Gamma0.inv %*% t(Om)
  S   <- rbind(cbind(D-CGCD %*% OmGOm , CGCD),
               cbind(-t(D.inv)%*%OmGOm, t(D.inv)))   # symplectic matrix
  Q <- schur.la(S)$vectors
  Xi  <-  Q[(n+1):(n+n),1:n] %*% solve(Q[1:n,1:n])
  Psi <- Gamma0 - H %*% Xi %*% t(H)
  K   <-  (Om - FF %*% Xi %*% t(H)) %*% solve(Psi) 
  browser()
  # check   shur(S)$schur = t(Q) %*% S  %*% Q is upper block triangular
  #  and   (eqn.A)  nxn 0 =
  #  Xi - (t(D)%*%solve(-t(H)%*%solve(Gamma0)%*%H +solve(Xi))%*%D + Om%*%solve(Gamma0) %*% t(Om)) 
  #  and   Woodbury identity  nxn 0 =
  #   solve(solve(Xi)-t(H)%*%solve(Gamma0)%*%H) - (Xi+Xi%*%t(H)%*%solve(Gamma0-H%*%Xi%*%t(H))%*%H%*% Xi) 
  #  and    (eqn.*) nxn 0=
  #    Xi-(t(D)%*%Xi%*%D + Om%*%solve(Gamma0)%*%t(Om) + t(D)%*%Xi%*%t(H)%*%solve(Psi)%*%H%*%Xi%*%D)
  #  and    (eqn.**) nxn 0=
  #    Xi-(FF%*%Xi%*%t(FF) + Om%*%solve(Psi)%*%t(Om) - 
  #          Om%*%solve(Psi)%*%H%*%Xi%*%t(FF) - 
  #          FF%*%Xi%*%t(H)%*%solve(Psi)%*%t(Om) + 
  #          FF%*%Xi%*%t(H)%*%solve(Psi)%*%H%*%Xi%*%t(FF))
  #  and nxn 0=solve(Psi)-solve(Gamma0-H%*%Xi%*%t(H))
  #  and nxn 0=solve(Psi)-(solve(Gamma0) -
  #           solve(Gamma0)%*%H%*%solve(t(H)%*%solve(Gamma0)%*%H -
  #           solve(Xi))%*%t(H)%*%solve(Gamma0))
  #  and nxn 0=Xi-(FF%*%Xi%*%t(FF) + 
  #        (Om-FF%*%Xi%*%t(H)) %*% solve(Psi) %*% t(Om-FF%*%Xi%*%t(H)))   (3.36)
  #  and nxn 0=Xi-(FF%*%Xi%*%t(FF) + 
  #     (Om-FF%*%Xi%*%t(H)) %*% solve(Gamma0-H%*%Xi%*%t(H)) %*% 
  #        t(Om-FF%*%Xi%*%t(H)))   (3.36)
  #  and # Aoki (3.33),  V&V (6)    nxn 0=
  #          Xi-(FF%*%Xi%*%t(FF) + K %*% Psi %*% t(K))   
  #  and # Aoki (3.34),  V&V (7)    pxp 0=
  #          Psi-Gamma0 + H%*%Xi%*%t(H)  
  #  and  Psi = innov. cov 
  #  and # Aoki (3.35),  V&V (3)    nxp 0=
  #         K %*% Psi - Om + FF%*%Xi%*%t(H)    
  z <-list(description="nested-balanced model a la Aoki",
                  F=FF,G=G,H=H,K= K)
  class(z) <-c("innov","SS","TSmodel")
  set.parameters(z)
}
#  start main program         # Form k block Hankel Matrix from M.
   p <- nrow(M)               # dim of endo. series
   if (is.null(m))            # dim of exogeneous series
     if(is.null(data))
       {cat("data or input dimension must be supplied to balance./n") 
        return("data or input dimension must be supplied to balance.") 
       }
     else m <- ncol(input.data(data))
   if (is.null(m)) m <- 0   # for case input.data(data) is null.
   r <- m + p       # each sub-matrix is p x r   (= p x (m+p) )
   k<- dim(M)[2] / r
   Hkl <- matrix(0, p*k, r*k) 
   for(i in 1:k)             # Hankel with 0s in SE half  
     Hkl[(1+p*(i-1)):(p*i), 1:(r*(1+k-i))]<- M[,(1+r*(i-1)):(r*k),drop=F]
#   k <- k %/% 2
#   Hkl <- matrix(0, p*k, r*k) 
#   for (i in 1:k)              # (smaller) completely filled Hankel
#      {for (j in 1:k)       
#       {Hkl[(1+p*(i-1)):(p*i), (1+r*(j-1)):(r*j)] <-M[ ,(1+r*(i+j-2)):(r*(i+j-1))]
#        }}
   svd.of.Hkl <- svd(Hkl)
   shifted.Hkl <- Hkl[,(r+1):dim(Hkl)[2]]
   shifted.Hkl <- cbind(shifted.Hkl,matrix(0,p*k,r))
   rtQ <- diag(sqrt(svd.of.Hkl$d),length(svd.of.Hkl$d))
   rtQ.inv <- diag(1/sqrt(svd.of.Hkl$d),length(svd.of.Hkl$d))   
   o.mtr <- svd.of.Hkl$u %*% rtQ 
   o.inv <- t(svd.of.Hkl$u %*% rtQ.inv )
   c.mtr <- rtQ %*% t(svd.of.Hkl$v)
   c.inv <- t(rtQ.inv %*% t(svd.of.Hkl$v))
 if (is.null(n))
  {svd.crit <- svd.criteria(svd.of.Hkl$d)
   if (!is.null(data))
     {n <- eval(parse(prompt="Enter the number of singular values to use: "))
      if( n>0) 
        {cat("Criteria test for state dimension 1 to ",n)
         values <- NULL 
         for (i in 1:n) 
           {cat("set Gamma0")
               browser()
               z <-set.model.Aoki(m,i,p, c.mtr,c.inv,o.mtr,o.inv,shifted.Hkl, Gamma0)
            z <- KF(z,data)
            z <-information.tests.calculations(z)
            values <-rbind(values, c(z))
            cat(".")
           }
        cat("\n")
        values<-cbind(values,svd.crit[1:n,])
        zz<-criteria.table.heading()
        options(width=120)
        print(values,digits=4)
        cat("opt     ")
        opt <-apply(values,2,order)[1,]  # minimum
        for (i in 1:(length(opt)-3)) cat(opt[i],"    ")
        cat("\n")
        zz<-criteria.table.legend()
        }
     }
   n <- eval(parse(prompt="Enter the state dimension (enter 0 to stop): "))         
  } 
   if( n<1) stop("TERMINATED! DIMENSION MUST BE GREATER THAN 0!")
   cat("set Gamma0")
   browser()
   set.model.Aoki(m,n,p, c.mtr,c.inv,o.mtr,o.inv,shifted.Hkl, Gamma0)
}

cov.Riccati.C <- function(FF,H, Gamma0, Om) 
{# solve Riccati eqn. with complex arrays. 

warning("This procedure has not been debugged or tested!")

 repeat
    {Gamma0.inv <- solve(Gamma0)
     D <- t(FF) - t(H) %*% Gamma0.inv %*% t(Om)   
     D.inv <- solve(D)
     CGCD <- t(H)%*% Gamma0.inv %*% H %*%Conj(t(D.inv))   # check?
     OmGOm <- Om %*% Gamma0.inv %*% t(Om)
     # symplectic matrix:
     S   <- rbind(cbind(D-CGCD %*% OmGOm , CGCD),
                  cbind(-Conj(t(D.inv))%*%OmGOm, Conj(t(D.inv)))) 
#    Q <- schurC.la(S)$vectors   #  S is not real !
     Q <- eigen(S)            #  S is not real !
     if( !any(1 ==Mod(Q$values))) break
     cat("Sympectic matrix has eigenvalues of mod=1. Augmenting Gamma0.\n")
     Gamma0 <- Gamma0 + diag(1e-1,dim(Gamma0)[1])
    }
   Q <- Q$vectors  #right.vectors
   n <- dim(FF)[1]
   Q[(n+1):(n+n),1:n] %*% solve(Q[1:n,1:n])  # soln. is not real
}
cov.Riccati.schur <- function(FF,H, Gamma0, Om)   # a la Aoki - this does NOT WORK.
{  Gamma0.inv <- solve(Gamma0)
   D <- t(FF) - t(H) %*% Gamma0.inv %*% t(Om)   
   D.inv <- solve(D)
   CGCD <- t(H)%*% Gamma0.inv %*% H %*%t(D.inv)
   OmGOm <- Om %*% Gamma0.inv %*% t(Om)
   # symplectic matrix:
   S   <- rbind(cbind(D-CGCD %*% OmGOm , CGCD),
               cbind(-t(D.inv)%*%OmGOm, t(D.inv)))  
   Q <- schur.la(S)$vectors
   n <- dim(FF)[1]
   Xi <- Q[(n+1):(n+n),1:n] %*% solve(Q[1:n,1:n])
   #  check 
   #  most importantly the special Riccati eqn. V&V (9), Aoki (3.36):  nxn 0=
   #  Xi-FF%*%Xi%*%t(FF) - (Om-FF%*%Xi%*%t(H)) %*% solve(Gamma0-H%*%Xi%*%t(H)) %*% t(Om-FF%*%Xi%*%t(H))
   #  alternately:    v <- svd(Gamma0-H%*%Xi%*%t(H))
   #  Xi-FF%*%Xi%*%t(FF) - (Om-FF%*%Xi%*%t(H)) %*% (v$u%*%diag(1/v$d)%*%t(v$v)) %*% t(Om-FF%*%Xi%*%t(H))
  #  and   Xi = t(Xi)
  #  and   schur.la(S)$schur = t(Q) %*% S  %*% Q is upper block triangular
  #  and   (eqn.A)  nxn 0 =
  #  Xi - (t(D)%*%solve(-t(H)%*%solve(Gamma0)%*%H +solve(Xi))%*%D + Om%*%solve(Gamma0) %*% t(Om)) 
  #  and   Woodbury identity  nxn 0 =
  #   solve(solve(Xi)-t(H)%*%solve(Gamma0)%*%H) - (Xi+Xi%*%t(H)%*%solve(Gamma0-H%*%Xi%*%t(H))%*%H%*% Xi) 

browser()
   Xi
}
schur.la       <- function # real schur decomposition of a real matrix A= U'SU-
   (a){
# 0 ==  max(abs(a-(result$vectors)%*%result$schur%*%t(result$vectors)))
# and result$schur is upper "block triangular"

warning("This procedure has not been debugged or tested!")

  n <- nrow(a)
  vs <-   matrix(0,n,n)
  storage.mode(a) <- "double"    #N.B. as.double messes up dim of result
  storage.mode(vs) <- "double"
  ans <-.Fortran("dgeesx", 
             as.logical(T),        # LJOBVS  
             as.logical(F),        # LSORT should be S.orN?. requires select
     #        as.logical  ("etext"),        # SELECT function
             as.integer(1),        # ISENSE
             as.integer(n),            # N
             a=a,                      # A    
             as.integer(n),            # LDA
             integer(1),               # SDIM
             wr=double(n),             # WR   
             wi=double(n),             # WI   
             vs=vs,                    # VS   
             as.integer(n),            # LDVS
             double(1),                 # RCONDE
             double(1),                 # RCONDV
             double(n*n),              # WORK
             as.integer(n*n),            # LWORK
             integer(n*n),             # IWORK
             as.integer(n*n),          # LIWORK 
             logical(n),               # BWORK
             info=integer(1) )         # INFO  20
# probably should check info here 
#browser()
  list(schur=ans$a, values=complex(real=ans$wr,imaginary=ans$wi), vectors=ans$vs, info=ans$info)
}
  
Cschur          <- function # complex schur decomposition of a real matrix - not working
   (a){

warning("This procedure has not been debugged or tested!")

  n <- nrow(a)
  wr <- wi <- d <- matrix(0, n, 1)
  q <- pm <- qq <- matrix(0,n,n)
  storage.mode(a) <- "double"    #N.B. as.double messes up dim of result
  storage.mode(q) <- "double"
  storage.mode(pm) <- "double"
  storage.mode(qq) <- "double"
  ans <- .Fortran("Cschur",
             a,
             as.integer(n),
             as.integer(n),
             as.double(wr),
             as.double(wi),
             q,
             pm,
             qq,
             as.double(d) )
  list(values=complex(real=ans[[4]],imaginary=ans[[5]]),
       orth=t(ans[[6]]), schur=ans[[6]] %*%a%*% t(ans[[6]]))
}

Hessenberg     <- function # decomposition of a square matrix A = Q'HQ, QQ'=I (so H=QAQ'), H is upper Hessenberg
   (a, tol= 1e-12){

warning("This procedure has not been debugged or tested!")

  n <- nrow(a)
  d <- matrix(0, n, 1)
  q  <- matrix(0,n,n)
  storage.mode(a) <- "double" 
  storage.mode(q) <- "double"
  storage.mode(d) <- "double"
  result <- .Fortran("orthpq",
             a,
             as.integer(n),
             as.integer(n),
             d,
             q,
             tol )
  list(H=result[[1]],Q=result[[5]])
}

#   Copyright 1993, 1994, 1995, 1996  Bank of Canada.
#   Copyright 1997 (June), Paul Gilbert.
#   Copyright 1997 (Aug.), Bank of Canada.
#   Copyright 1998, Bank of Canada.

#   The user of this software has the right to use, reproduce and distribute it.
#   Bank of Canada makes no warranties with respect to the software or its 
#   fitness for any particular purpose. The software is distributed by the Bank
#   of Canada and by Paul Gilbert solely on an "as is" basis. By using the  
#   software, user agrees to accept the entire risk of using this software.

################################################################################



# For installation instructions see the file read.me or the brief user's
#    guide (postscipt file guide.ps).

#######################################################################

#    test functions for examples in the Brief User's Guide   <<<<<<<<<<

#######################################################################

guide.example.tests.part1 <- function( verbose=T, synopsis=T, fuzz.small=1e-14, fuzz.large=1e-8, graphics=T, pause=F)
{# test examples in Brief User's guide
 # NOTE: it was necessary to reduce fuzz from 1e-14 because of differences
 # in the results between Splus 3.2 and Splus 3.3 (C libraries were changed).
 # Differences affected lsfit (used in est.VARX.ls) among other things.


  # If no device is active then write to postscript file 
  if (graphics)
   {if (!exists.graphics.device())
      {postscript(file="zot.postscript.test.ps",width=6,height=6,pointsize=10,
                   onefile=F, print.it=F, append=F)
       on.exit((function()
             {dev.off(); synchronize(1); rm("zot.postscript.test.ps")})())
      }
    else
      {old.par <- par()
       on.exit(par(old.par))
      }
    if(pause) dev.ask(ask=T)
   }


  max.error <- NA
  if (synopsis & !verbose) cat("All Brief User Guide example part 1 tests ...")
  if (verbose) cat("Guide part 1 test 1 ... ")
  # previously search() was used to determine "from", but DSE.HOME is better
  #  if(1==pmatch("MS Windows",version$os, nomatch=0))
  #     from <- (search()[grep("b*/DSE/_Data", search())])[1]
  #  else
  #     from <- (search()[grep("b*/DSE/.Data", search())])[1]
  # if DSE is not in the search path then data file is assumed to be in 
  #  the present working directory
  #  if(5 < nchar(from)) from<-substring(from, first=1, last = nchar(from) -5)
  #  from <- paste(from,"eg1.dat", sep="")

  from <- paste(DSE.HOME, "/data/eg1.dat", sep="")
  data <- t(matrix(scan(from), 5,364))[,2:5]
  #  data <- list(input=ts(data[,1  ,drop=F], start=c(1961,3), frequency=12),
  #              output=ts(data[,2:4,drop=F], start=c(1961,3), frequency=12))
  data <- list(input=tframed(data[,1  ,drop=F],
                                         list(start=c(1961,3), frequency=12)),
              output=tframed(data[,2:4,drop=F], 
                                         list(start=c(1961,3), frequency=12)))
  class(data) <- "TSdata"
  input.series.names(data)   <-  "u1"
  output.series.names(data) <-  c("y1","y2","y3")
  error <- abs(126943980.50000011921 - sum(output.data(data)))
  ok <- fuzz.small > error
  if (!ok) {if (is.na(max.error)) max.error <- error
            else max.error <- max(error, max.error)}
  all.ok <- ok 
  if (verbose) 
    {if (ok) cat("ok\n")
     else    cat("failed! (error magnitude= ", error,")\n")
    }

  if (verbose) cat("Guide part 1 test 2 ... ")
  model1 <- est.VARX.ls(data, warn=F)
  model2 <- est.SS.Mittnik(data, n=14)
#  summary(model1)
#  summary(model2)
#  print(model1)
#  print(model2)
#  stability(model1)
#  stability(model2)
   if (graphics) plot(model1)

  error <- max(Mod(c(15.430979953081722655   - sum(TSmodel(model1)$A),
                         -1.1078692933906153506  - sum(TSmodel(model2)$F),
                         2.4561249653768193468   - sum(roots(model2)) )))
#                        2.4561249653768193468+0i- sum(roots(model2)) )))
  ok <- fuzz.large > error
  if (!ok) {if (is.na(max.error)) max.error <- error
            else max.error <- max(error, max.error)}
  ok <-  ok & is.TSestModel(model1) & is.TSestModel(model2)
  all.ok <- all.ok & ok 
  if (verbose) 
    {if (ok) cat("ok\n")
     else    cat("failed! (error magnitude= ", error,")\n")
    }

  if (verbose) cat("Guide part 1 test 3 ... ")
  ar<-array(c(1,.5,.3,0,.2,.1,0,.2,.05,1,.5,.3),c(3,2,2))
  ma<-array(c(1,.2,0,.1,0,0,1,.3),c(2,2,2))
  arma<-list(A=ar,B=ma,C=NULL)
  class(arma)<-c("ARMA","TSmodel")
  arma<-set.parameters(arma)
#  print(arma)
  ok <- is.TSmodel(arma) 
  all.ok <- all.ok & ok 
  if (verbose) 
    {if (ok) cat("ok\n")
     else    cat("failed!\n")
    }

  if (verbose) cat("Guide part 1 test 4 ... ")
  data.arma.sim<-simulate(arma)
  arma<-l(arma,data.arma.sim)
#  summary(arma)
  if (graphics) 
     {plot(data.arma.sim)
      plot(arma)
     }
  ok <- is.TSdata(data) & is.TSestModel(arma) 
  all.ok <- all.ok & ok 
  if (verbose) 
    {if (ok) cat("ok\n")
     else    cat("failed!\n")
    }

  if (verbose) cat("Guide part 1 test 5 ... ")
  f<-array(c(.5,.3,.2,.4),c(2,2))
  h<-array(c(1,0,0,1),c(2,2))
  k<-array(c(.5,.3,.2,.4),c(2,2))
  ss<-list(F=f,G=NULL,H=h,K=k)
  class(ss)<-c("innov","SS","TSmodel")
  ss<-set.parameters(ss)
#  ss
  ok <- is.SS(ss)
  all.ok <- all.ok & ok
  if (verbose) 
    {if (ok) cat("ok\n")
     else    cat("failed!\n")
    }

  if (verbose) cat("Guide part 1 test 6 ... ")
  data.ss.sim<-simulate(ss)
  ss<-l(ss,data.ss.sim)
#  summary(ss)
  if (graphics) plot(ss)
  ok <- is.TSestModel(ss)
  all.ok <- all.ok & ok
  if (verbose) 
    {if (ok) cat("ok\n")
     else    cat("failed!\n")
    }

  if (verbose) cat("Guide part 1 test 7 ... ")
  ss.from.arma <- l(to.SS(arma), data.arma.sim)
  arma.from.ss <- l(to.ARMA(ss), data.ss.sim)
#  summary(ss.from.arma)
#  summary(arma)
#  summary(arma.from.ss)
#  summary(ss)
#  stability(arma)
#  stability(ss.from.arma)
#  caution: tests on $estimates will depend on seed when data is generated.
  error <- max(Mod(c(-0.15000000000000018874 - sum(TSmodel(ss.from.arma)$F),
                         0.47999999999999998224  - sum(TSmodel(arma.from.ss)$A),
                         -1                      - sum(roots(ss.from.arma)) )))
#                        -1+0i                   - sum(roots(ss.from.arma)) )))
  ok <- fuzz.small > error
  if (!ok) {if (is.na(max.error)) max.error <- error
            else max.error <- max(error, max.error)}
  all.ok <- all.ok & ok 
  if (verbose) 
    {if (ok) cat("ok\n")
     else    cat("failed! (error magnitude= ", error,")\n")
    }

  if (synopsis) 
    {if (verbose) cat("All Brief User Guide example part 1 tests completed")
     if (all.ok) cat(" ok\n")
     else    cat(", some failed! (error magnitude= ", max.error,")\n")
    }
  invisible(all.ok)
}




guide.example.tests.part2 <- function( verbose=T, synopsis=T, fuzz.small=1e-14, fuzz.large=1e-8, graphics=T, pause=F)
{# test examples in Brief User's guide
  if (synopsis & !verbose) cat("Brief User Guide example part 2 tests ...")
  if (verbose) cat("Guide part 2 test 1 ... ")

  # If no device is active then write to postscript file 
  if (graphics)
   {if (!exists.graphics.device())
      {postscript(file="zot.postscript.test.ps",width=6,height=6,pointsize=10,
                   onefile=F, print.it=F, append=F)
       on.exit((function()
                {dev.off(); synchronize(1); rm("zot.postscript.test.ps")})())
      }
    else
      {old.par <- par()
       on.exit(par(old.par))
      }
    if(pause) dev.ask(ask=T)
   }

  max.error <- NA
  all.ok <- T
  if (is.S())
    {test.seed1 <- c(13,44,1,25,56,0,6,33,22,13,13,0)
     test.seed2 <- c(13,43,7,57,62,3,30,29,24,54,47,2)
     test.seed4 <- c(29,55,47,18,33,1,15,15,34,46,13,2)
     test.seed3 <- c( 53,41,26,39,10,1,19,25,56,32,28,3)
    }
  else if (is.R()) 
    {test.seed1 <- test.seed2 <- test.seed3 <- test.seed4 <- c(979, 1479, 1542)}

  from <- paste(DSE.HOME, "/data/eg1.dat", sep="")
  eg1.DSE.data <-example.get.eg.raw.data(from) #retrieves data from file
  eg2.DSE.data.names <- list(
	output=list(series=c( "I37005"), 
 	server=c("ets")), 
 	output.names=c( "manuf.prod.") )
  class(eg2.DSE.data.names) <- c("TSPADIdata", "TSdata")
# Fame call disabled for testing:  eg2.DSE.data <- TSdata(eg2.DSE.data.names)
  eg3.DSE.data.names <- list(
 	input=list(series="lfsa455", server= "ets"),
 	input.transformations= "percent.change.vector",
 	input.names= "manuf.emp.",
 	output=list(series=c( "i37005"), 
	server=c("ets")), 
 	output.names=c( "manuf.prod."),
 	output.transformations= c("percent.change.vector"),
 	pad.start=F,
 	pad.end =T 
 	)
  class(eg3.DSE.data.names) <- c("TSPADIdata", "TSdata")
# Fame call disabled for testing: eg3.DSE.data <- TSdata(eg3.DSE.data.names)

 JofF.VAR.data.names <- list(
	input = list(series= "B14017", server="ets"),
	input.transformations= "diff",
	input.names=c("R90"),
	output = list(series=c("P484549", "I37026", "b1627", "b14013",
				 "b4237", "D767608", "b3400", "M.BCPI",
				 "M.JQIND", "M.CUSA0"),
		      server= "ets"),
	output.transformations=c("percent.change.vector", 
			"percent.change.vector","percent.change.vector",
			"diff", "diff", "percent.change.vector",
			"percent.change.vector", "percent.change.vector",
			"percent.change.vector", "percent.change.vector"),
	output.names=c("CPI", "GDP", "M1", "RL", "TSE300", 
			"employment", "PFX", "com. price ind.", 
			"US ind. prod.", "US CPI")
	)
  class(JofF.VAR.data.names) <- c("TSPADIdata", "TSdata")
# Fame call disabled for testing: JofF.VAR.data <- TSdata(JofF.VAR.data.names)
  error <- abs(3352.4721630925987483 - 
            sum(c(output.data(JofF.VAR.data.1dec93),
                   input.data(JofF.VAR.data.1dec93))))
  ok <-  fuzz.small > error
  if (!ok) {if (is.na(max.error)) max.error <- error
            else max.error <- max(error, max.error)}
  all.ok <- all.ok &ok 
  if (verbose) 
    {if (ok) cat("ok\n")
     else    cat("failed! (error magnitude= ", error,")\n")
    }

# Section 4 - Model Estimation

  if (verbose) cat("Guide part 2 test 2 ... ")
  model.eg1.ls <- est.VARX.ls(trim.na(eg1.DSE.data), warn=F)
  opts <-options(warn=-1) 
    subsample.data <- window(eg1.DSE.data, start=c(1972,1), end=c(1992,12))
  options(opts)
  # summary(model.eg1.ls)
  # print(model.eg1.ls)
  if (graphics)
    {plot(model.eg1.ls)
     plot(model.eg1.ls, Start=c(1990,1))
    }
  z <- check.residuals(model.eg1.ls, Plot=F, pac=T)
  if (is.S())      check.value <-
                  c(4.67445135116577148, 3274.42578125, -2371.9997808950302)
  else if (is.R()) check.value <- 
                  c(4.678440020271987,      0.0,        -2371.999780895033837)
  error <- max(abs(check.value - c(sum(z$acf),sum(z$pacf),sum(z$cusum)) ))
  ok <-  fuzz.large > error 
  if (!ok) {if (is.na(max.error)) max.error <- error
            else max.error <- max(error, max.error)}
  all.ok <- all.ok & ok 
  if (verbose) 
    {if (ok) cat("ok\n")
     else    {cat("failed! (error magnitude= ", error,")\n")
              print(c(sum(z$acf),sum(z$pacf),sum(z$cusum)), digits=18)
             }
    }

  if (verbose) cat("Guide part 2 test 3 ... ")
  # NB- non-stationary data. ar is not really valid
  model.eg1.ar <- est.VARX.ar(trim.na(eg1.DSE.data), warn=F) 
  model.eg1.ss <- est.SS.from.VARX(trim.na(eg1.DSE.data), warn=F) 
# model.eg1.mle <- est.max.like(trim.na(eg1.DSE.data),model.eg1.ar) # this may be slow
  if (is.S())      check.value <-
                  c(6738.642100450848, 6921.352363629516)
  else if (is.R()) check.value <- 
                  c(6921.352363629515, 6921.352363629516) #using ls for ar
  error <- max(abs(check.value -
             c(model.eg1.ar$estimates$like[1],model.eg1.ss$estimates$like[1])))
  ok <- fuzz.small > error
  if (!ok) {if (is.na(max.error)) max.error <- error
            else max.error <- max(error, max.error)}
  all.ok <- all.ok & ok 
  if (verbose) 
    {if (ok) cat("ok\n")
     else    cat("failed! (error magnitude= ", error,")\n")
    }

  if (verbose) cat("Guide part 2 test 4a... ")
   eg4.DSE.data<- JofF.VAR.data.1dec93
   output.data(eg4.DSE.data) <- output.data(eg4.DSE.data, series=c(1,2,6,7))
 # following is optional 
 # tframe(output.data(eg4.DSE.data))<- tframe(output.data(JofF.VAR.data.1dec93))

  model.eg4.bb <- est.black.box(trim.na(eg4.DSE.data), max.lag=3, verbose=F) 
  error <- abs(614.70500313590287078 - model.eg4.bb$estimates$like[1] )
  ok <-  fuzz.large > error 
  all.ok <- all.ok & ok 
  if (verbose) 
    {if (ok) cat("ok\n")
     else    cat("failed! (error magnitude= ", error,")\n")
    }

  if (verbose) cat("Guide part 2 test 4b... ")
  z <- information.tests(model.eg1.ar, model.eg1.ss, Print=F)
  if (is.S())      check.value <- 231152.464267979725
  else if (is.R()) check.value <- 233876.86196027516   #using ls for ar
  error <- abs(check.value - sum(z[!is.na(z)]) )
  ok <-  fuzz.small > error 
  if (!ok) {if (is.na(max.error)) max.error <- error
            else max.error <- max(error, max.error)}
  all.ok <- all.ok & ok 
  if (verbose) 
    {if (ok) cat("ok\n")
     else    cat("failed! (error magnitude= ", error,")\n")
    }

# Section 5 - Forecasting

  if (verbose) cat("Guide part 2 test 5 ... ")
  eg4.DSE.model <- est.VARX.ls(eg4.DSE.data)
#  Fame call disabled for testing: new.data <- TSdata(eg4.DSE.data.names) 
  new.data <- list(
              input= tframed(rbind(input.data(eg4.DSE.data), matrix(.1,10,1)), 
                       list(start=start(eg4.DSE.data),
                       frequency=frequency(eg4.DSE.data))),    
              output=tframed(rbind(output.data(eg4.DSE.data),matrix(.3,5,4)), 
                       list(start=start(eg4.DSE.data),
                       frequency=frequency(eg4.DSE.data))))
  class(new.data) <- "TSdata"
  series.names(new.data) <- series.names(eg4.DSE.data)
  z  <- l(TSmodel(eg4.DSE.model), trim.na(new.data)) 
#  z <- l(TSmodel(eg4.DSE.model), trim.na(TSdata(eg4.DSE.data.names)))
  error <- max(abs(556.55870662521476788 -z$estimates$like[1]))
  ok <-  fuzz.large > error 
  if (!ok) {if (is.na(max.error)) max.error <- error
            else max.error <- max(error, max.error)}
  all.ok <- all.ok & ok 
  if (verbose) 
    {if (ok) cat("ok\n")
     else    cat("failed! (error magnitude= ", error,")\n")
    }

  if (verbose) cat("Guide part 2 test 6 ... ")
  zz <- forecast(TSmodel(eg4.DSE.model), new.data)
  z <-  forecast(TSmodel(eg4.DSE.model), trim.na(new.data), 
		conditioning.inputs=input.data(new.data))
  if (graphics) plot(zz, Start=c(1990,6))
  error <- abs(4.7990339556773520258 - sum(z$forecast[[1]]))
  ok <- fuzz.small > error
  if (!ok) {if (is.na(max.error)) max.error <- error
            else max.error <- max(error, max.error)}
  ok <- test.equal(zz,z) & ok
  all.ok <- all.ok & ok 
  if (verbose) 
    {if (ok) cat("ok\n")
     else    cat("failed! (error magnitude= ", error,")\n")
    }

  if (verbose) cat("Guide part 2 test 7 ... ")
  z <- forecast(eg4.DSE.model, conditioning.inputs.forecasts=matrix(.5,6,1)) 
  # Fame call disabled for testing: 
  # z <- forecast(TSmodel(eg4.DSE.model), TSdata(JofF.VAR.data.names), 
  #		conditioning.inputs.forecasts=matrix(.5,6,1))
  # summary(z)
  # print(z)
  if (graphics)
    {plot(z)
     plot(z, Start=c(1990,1))
    }
  #  z$forecast[[1]]
  #  window(z$forecast[[1]], start=c(1994,5)) 
  ok <- all(start(eg4.DSE.model$data)   == c(1974,2)) &
        all(start(JofF.VAR.data.1dec93) == c(1974,2)) 
  error <- max(abs(c(5.9414711908521793404 - sum(z$forecast[[1]][1:6,]),
                  3.7410224783909828972 - 
                     sum(window(z$forecast[[1]], start=c(1993,12), warn=F)))))
  ok <-  ok & (fuzz.small > error) 
  if (!ok) {if (is.na(max.error)) max.error <- error
            else max.error <- max(error, max.error)}
  all.ok <- all.ok & ok 
  if (verbose) 
    {if (ok) cat("ok\n")
     else    cat("failed! (error magnitude= ", error,")\n")
    }

  if (verbose) cat("Guide part 2 test 8 ... ")
  z <- l(TSmodel(eg4.DSE.model), new.data)
# cat(" ") #Rbug cat here seems to flush something and avoid a seg fault crash
  if (graphics) plot(z)
  z <- feather.forecasts(TSmodel(eg4.DSE.model), new.data)
# cat(" ") #Rbug cat here seems to flush something and avoid a seg fault crash
  if (graphics) plot(z)
  zz <-feather.forecasts(TSmodel(eg4.DSE.model), new.data,
                          from.periods =c(20,50,60,70,80), horizon=150)
# cat(" ") #Rbug cat here seems to flush something and avoid a seg fault crash
  if (graphics) plot(zz)
  error <- max(abs(c(54.838475604100473504 -
                           sum( z$feather.forecasts[[1]][10:46,]),
                       53.824873541066445171 - 
                           sum(zz$feather.forecasts[[5]][80:116,]))))
  ok <-fuzz.large > error
  if (!ok) {if (is.na(max.error)) max.error <- error
            else max.error <- max(error, max.error)}
  all.ok <- all.ok & ok 
  if (verbose) 
    {if (ok) cat("ok\n")
     else    cat("failed! (error magnitude= ", error,")\n")
    }

  if (verbose) cat("Guide part 2 test 9 ... ")
  z <- horizon.forecasts(TSmodel(eg4.DSE.model), new.data, horizons=c(1,3,6))
  if (graphics) plot(z)
  error <- abs(653.329319170802592 - sum(z$horizon.forecasts) )
  ok <-  fuzz.large > error 
  if (!ok) {if (is.na(max.error)) max.error <- error
            else max.error <- max(error, max.error)}
  all.ok <- all.ok & ok 
  if (verbose) 
    {if (ok) cat("ok\n")
     else    cat("failed! (error magnitude= ", error,")\n")
    }

  if (verbose) cat("Guide part 2 test 10... ")
  fc1 <- forecast.cov(TSmodel(eg4.DSE.model), data=eg4.DSE.data)
  if (graphics) 
    {plot(fc1)
     plot(forecast.cov(TSmodel(eg4.DSE.model), data=eg4.DSE.data, horizons= 1:4)) 
    }
  fc2 <- forecast.cov(TSmodel(eg4.DSE.model), data=eg4.DSE.data, zero=T, trend=T)
  if (graphics) plot(fc2)
  error <- max(abs(c(14.933660144821400806 - sum(fc1$forecast.cov[[1]]),
                        14.933660144821400806 - sum(fc2$forecast.cov[[1]]),
                        31.654672476928297442 - sum(fc2$forecast.cov.zero),
                        18.324461923341953451 - sum(fc2$forecast.cov.trend) )))
  ok <- fuzz.small > error
  if (!ok) {if (is.na(max.error)) max.error <- error
            else max.error <- max(error, max.error)}
  if (is.na(ok)) ok <- F
  all.ok <- all.ok & ok 
  if (verbose) 
    {if (ok) cat("ok\n")
     else    cat("failed! (error magnitude= ", error,")\n")
    }

  if (verbose) cat("Guide part 2 test 11... ")
  mod1 <- set.parameters(
 	TSmodel(list(A=array(c(1,-.25,-.05), c(3,1,1)), B=array(1,c(1,1,1)))) )
  mod2 <- set.parameters(
 	TSmodel(list(A=array(c(1,-.8, -.2 ), c(3,1,1)), B=array(1,c(1,1,1)))) )
  mod3 <- set.parameters( TSmodel(list(
 	A=array(c( 
 	1.00,-0.06,0.15,-0.03,0.00,0.02,0.03,-0.02,0.00,-0.02,-0.03,-0.02,
	0.00,-0.07,-0.05,0.12,1.00,0.20,-0.03,-0.11,0.00,-0.07,-0.03,0.08,
 	0.00,-0.40,-0.05,-0.66,0.00,0.00,0.17,-0.18,1.00,-0.11,-0.24,-0.09 )
		,c(4,3,3)), 
 	B=array(diag(1,3),c(1,3,3)))) )
  e.ls.mod1 <- eval.estimation( mod1, replications=100, 
 	seed=test.seed1,
 	simulation.args=list(sampleT=100, sd=1), 
 	estimation="est.VARX.ls", estimation.args=list(max.lag=2), 
 	criterion="TSmodel", quiet=T)

#    e.ar.mod1 <- eval.estimation( mod1, replications=100, 
#   	seed=test.seed1,
#   	simulation.args=list(sampleT=100, sd=1), 
#   	estimation="est.VARX.ar", estimation.args=list(max.lag=2, aic=F), 
#   	criterion="TSmodel", quiet=T)
#   plot(parms(e.ar.mod1))


  if (is.S())      check.value <- -0.29855874505752699744
  else if (is.R()) check.value <- -0.3699580622977685
  error <- abs( check.value - sum(parms(e.ls.mod1$result[[100]])))
  ok <- fuzz.small > error
  if (!ok) {if (is.na(max.error)) max.error <- error
            else max.error <- max(error, max.error)}
  all.ok <- all.ok & ok 
  if (verbose) 
    {if (ok) cat("ok\n")
     else    {cat("failed! (error magnitude= ", error,")\n")
print(sum(parms(e.ls.mod1$result[[100]])), digits=18)
             }
    }

  if (verbose) cat("Guide part 2 test 12... ")
  e.ls.mod2 <- eval.estimation( mod2, replications=100, 
                     seed=test.seed2,
                     simulation.args=list(sampleT=100, sd=1), 
                     estimation="est.VARX.ls", estimation.args=list(max.lag=2), 
                     criterion="TSmodel", quiet=T)
  if (graphics)
    {par(mfcol=c(2,1)) # set the number of plots on the plotics device
     plot(parms(e.ls.mod1))
     plot(parms(e.ls.mod2)) 
     par(mfcol=c(2,1)) # set the number of plots on the plotics device
     plot(parms(e.ls.mod1), cum=F, bounds=F) 
     plot(parms(e.ls.mod2), cum=F, bounds=F) 
     distribution(parms(e.ls.mod1), bandwidth=.2)
     distribution(parms(e.ls.mod2), bandwidth=.2)
    } 

  if (is.S())      check.value <- -1.0021490287427212706
  else if (is.R()) check.value <- -1.0028944627996932
  error <- abs(check.value - sum(parms(e.ls.mod2$result[[100]])))
  ok <- fuzz.small > error
  if (!ok) {if (is.na(max.error)) max.error <- error
            else max.error <- max(error, max.error)}
  all.ok <- all.ok & ok 
  if (verbose) 
    {if (ok) cat("ok\n")
     else  {  cat("failed! (error magnitude= ", error,")\n")
print(sum(parms(e.ls.mod2$result[[100]])), digits=18)
             }
    }

  if (verbose) cat("Guide part 2 test 13... ")
  e.ls.mod1.roots <- roots(e.ls.mod1)
  if (graphics)
    {plot(e.ls.mod1.roots) 
     plot(e.ls.mod1.roots, complex.plane=F)
     plot(roots(e.ls.mod2), complex.plane=F) 
     distribution(e.ls.mod1.roots, bandwidth=.2) 
     distribution(roots(e.ls.mod2), bandwidth=.1) 
    }

  if (is.S())      check.value <- 0.36159459310761993267
  else if (is.R()) check.value <- 0.211967720656464
# error <- Mod(0.36159459310761993267+0i -sum(e.ls.mod1.roots$result[[100]]))
  error <- Mod(check.value   -sum(e.ls.mod1.roots$result[[100]]))
  ok <- fuzz.small > error
  if (!ok) {if (is.na(max.error)) max.error <- error
            else max.error <- max(error, max.error)}
  all.ok <- all.ok & ok 
  if (verbose) 
    {if (ok) cat("ok\n")
     else   { cat("failed! (error magnitude= ", error,")\n")
print(sum(e.ls.mod1.roots$result[[100]]), digits=18)
             }
    }

  if (verbose) cat("Guide part 2 test 14... ")
  pc <- forecast.cov.estimators.wrt.true(mod3,
 	seed=test.seed3,
 	estimation.methods=list(est.VARX.ls=list(max.lag=6)),
 	est.replications=2, pred.replications=10, quiet=T)
  # the fuzz.small has to be relaxed here to accomodate differences in rnorm
  #   between Splus3.1 and Splus3.2  (the numbers are from Splus3.2)

  if      (is.S())  check.value <-
      c(60.927013860328429473, 62.32729288591478678, 63.17808145947956433) 
  else if (is.R()) check.value <-
      c(54.164759056117525,    53.51929727783967,    59.34152615941156 )
  error <- max(abs(check.value -c(sum(pc$forecast.cov[[1]]), 
                      sum(pc$forecast.cov.zero), sum(pc$forecast.cov.trend) )))
  ok <- fuzz.large > error
  if (!ok) {if (is.na(max.error)) max.error <- error
            else max.error <- max(error, max.error)}
  all.ok <- all.ok & ok 
  if (verbose) 
    {if (ok) cat("ok\n")
     else  { cat("failed! (error magnitude= ", error,")\n")
            print(sum(pc$forecast.cov[[1]]),  digits=18)
            print(sum(pc$forecast.cov.zero),  digits=18)
            print(sum(pc$forecast.cov.trend), digits=18)
           }
    }

  if (verbose) cat("Guide part 2 test 15... ")
  pc.rd <- forecast.cov.reductions.wrt.true(mod3,
 	seed=test.seed4,
 	estimation.methods=list(est.VARX.ls=list(max.lag=3)),
 	est.replications=2, pred.replications=10, quiet=T)

  if (is.S())      check.value <-
         c(58.75543799264762157,60.451513998215133938, 64.089618782185240775) 
  else if (is.R()) check.value <- 
         c(51.237201863944890,  53.51929727783967,     59.34152615941156 )
  error <- max(abs(check.value - c(sum(pc.rd$forecast.cov[[1]]),
                  sum(pc.rd$forecast.cov.zero), sum(pc.rd$forecast.cov.trend))))
  ok <- fuzz.large > error
  if (!ok) {if (is.na(max.error)) max.error <- error
            else max.error <- max(error, max.error)}
  all.ok <- all.ok & ok 
  if (verbose) 
    {if (ok) cat("ok\n")
     else   { cat("failed! (error magnitude= ", error,")\n")
              print(sum(pc.rd$forecast.cov[[1]]),  digits=18)
              print(sum(pc.rd$forecast.cov.zero),  digits=18)
              print(sum(pc.rd$forecast.cov.trend), digits=18)
           }
    }

  if (verbose) cat("Guide part 2 test 16... ")
  z <-out.of.sample.forecast.cov.estimators.wrt.data(trim.na(eg1.DSE.data),
 	estimation.sample=.5,
 	estimation.methods = list(
 		est.VARX.ar=list(warn=F), 
 		est.VARX.ls=list(warn=F)), 
 	trend=T, zero=T)
  if (graphics) plot(z)
  opts <- options(warn=-1)
  zz <-out.of.sample.forecast.cov.estimators.wrt.data(trim.na(eg1.DSE.data),
 	estimation.sample=.5,
 	estimation.methods = list(
 		est.black.box4=list(max.lag=3, verbose=F, warn=F),
		est.VARX.ls=list(max.lag=3, warn=F)), 
	trend=T, zero=T)

    zf<-horizon.forecasts(zz$multi.model[[1]],zz$data, horizons=c(1,3,6))
  options(opts)
  zf<- zf$horizon.forecasts[3,30,]
  if (graphics) plot(z)
  if (is.S())      check.value <- 
     c(6120.97621905043979, 175568.040899036743, 24.568074094041549,
       1e-5*c(158049871127.845642, 3330592793.50789356, 
              1242727188.69001055, 1606263575.00784183))

  else if (is.R()) 
    {check.value <-
       c(6120.9762190509673, 175568.04089903546, 24.568074093999403,
       1e-5*c(3330592793.507894, 3330592793.507894,
              1242727188.690011, 1606263575.0078418)) # using ls for ar
    }
  error <- max(abs(check.value -
         c(zf,  1e-5*c(sum( z$forecast.cov[[1]]), sum( z$forecast.cov[[2]]),
                       sum(zz$forecast.cov[[1]]), sum(zz$forecast.cov[[2]])))))
  ok <-  fuzz.large > error
  if (!ok) {if (is.na(max.error)) max.error <- error
            else max.error <- max(error, max.error)}
  all.ok <- all.ok & ok 
  if (verbose) 
    {if (ok) cat("ok\n")
     else  {cat("failed! (error magnitude= ", error,")\n")
            print(zf,digits=18)
            print(sum(z$forecast.cov[[1]]),digits=18)
            print(sum(z$forecast.cov[[2]]),digits=18)
            print(sum(zz$forecast.cov[[1]]),digits=18)
            print(sum(zz$forecast.cov[[2]]),digits=18)
           }
    }

  if (synopsis) 
    {if (verbose) cat("All Brief User Guide example tests part 2 completed")
     if (all.ok) cat(" ok\n")
     else    cat(", some failed! (max. error magnitude= ", max.error,")\n")
    }
  invisible(all.ok)
}


guide.example.tests.part3 <- function( verbose=T, synopsis=T, fuzz.small=1e-14, fuzz.large=1e-8, graphics=T, pause=F)
{
 if (!exists("check.padi.server"))
    stop("This test requires a PADI server called ets and is primarily intended for checks at the BOC. The code can be examined as an example.")
 else  if (!check.padi.server("bc")) # ets
    stop("This test requires a server called ets and is primarily intended for checks at the BOC. The code can be examined as an example.")

  # If no device is active then write to postscript file 
     if (graphics)
   {if (!exists.graphics.device())
      {postscript(file="zot.postscript.test.ps",width=6,height=6,pointsize=10,
                   onefile=F, print.it=F, append=F)
       on.exit((function()
                {dev.off(); synchronize(1); rm("zot.postscript.test.ps")})())
      }
    else
      {old.par <- par()
       on.exit(par(old.par))
      }
    if(pause) dev.ask(ask=T)
   }

  max.error <- NA
  if (synopsis & !verbose) cat("Brief User Guide example part 3 tests ...")
  if (verbose) cat("Guide part 3 test 1 ... ")

# help.start.DSE(browser="mosaic")
# help.start.DSE()
# openlook()  #for plot window


# manufacturing

#  two outputs, no input

   cbps.manuf.data2.ids <- list(
      output=list(c("ets", "", "i37013","percent.change.vector","cbps.prod."),
                  c("ets", "", "i37005","percent.change.vector","manuf.prod.")),
      pad.start=F,
      pad.end =T )
   class(cbps.manuf.data2.ids) <- c("TSPADIdata", "TSdata")
   # get data
   cbps.manuf.data2 <- TSdata(cbps.manuf.data2.ids)


   # Estimate models with employment as input (exogenous) variable.

   manuf.data.ids <- list(
      input =list(c("ets", "","lfsa455","percent.change.vector","manuf.emp.")),
      output=list(c("ets", "", "I37005","percent.change.vector","manuf.prod.")),
      pad.start=F,
      pad.end =T  )  #pad.end=F for estimation or use trim.na(data) 
   class(manuf.data.ids) <- c("TSPADIdata", "TSdata")
   manuf.data <- TSdata(manuf.data.ids)
   if (graphics) plot(manuf.data)       # shows NA data in the middle
   manuf.data <- window(manuf.data, start=c(1976,2))
   manuf.data.ids$start <- c(1976,2)  # avoid NAs in middle as with window above
   manuf.data <- TSdata(manuf.data.ids)

   if (graphics) plot(window(manuf.data, start=c(1995,11)))   
   #      if (graphics) plot(manuf.data, Start=c(1995,11))     Bug !!!
   # cbps
   #  two inputs
   cbps.manuf.data.ids <- list(
      input =list(c("ets", "","lfsa462","percent.change.vector","cbps.emp."),
                  c("ets", "","lfsa455","percent.change.vector","manuf.emp.")),
      output=list(c("ets", "", "i37013","percent.change.vector","cbps.prod.")),
      pad.start=F,
      pad.end =T  )
   class(cbps.manuf.data.ids) <- c("TSPADIdata", "TSdata")
   cbps.manuf.data.ids$start <- c(1976,2)  # avoid NAs in middle
   cbps.manuf.data <- TSdata(cbps.manuf.data.ids)
   # cbps.manuf.data <- window(cbps.manuf.data, start=c(1976,2))

#  two outputs, one input

   cbps.manuf.data3.ids <- list(
      input =list(c("ets", "","lfsa462","percent.change.vector","cbps.emp.")),
      output=list(c("ets", "", "i37013","percent.change.vector","cbps.prod."),
                  c("ets", "", "i37005","percent.change.vector","manuf.prod.")),
      pad.start=F,
      pad.end =T   )
   class(cbps.manuf.data3.ids) <- c("TSPADIdata", "TSdata")
   cbps.manuf.data3.ids$start <- c(1976,2)  # avoid NAs in middle
   cbps.manuf.data3 <- TSdata(cbps.manuf.data3.ids)
   # cbps.manuf.data3 <- window(cbps.manuf.data3, start=c(1976,2))

  ok <- is.TSdata(manuf.data) & is.TSdata(cbps.manuf.data) & 
        is.TSdata(cbps.manuf.data3 )
  if (!ok) {if (is.na(max.error)) max.error <- error
            else max.error <- max(error, max.error)}
  all.ok <- ok 
  if (verbose) 
    {if (ok) cat("ok\n")
     else    cat("failed! (error magnitude= ", error,")\n")
    }

  if (verbose) cat("Guide part 3 test 2 ... ")

# estimate model

# dev.ask(T)

# manuf.model  <- bft(trim.na(manuf.data)) # might take some time
# or
# manuf.model  <- bft(trim.na(manuf.data), max.lag=5) 
# or
manuf.model  <- bft(trim.na(manuf.data), verbose=F, max.lag=5)
# may give "Warning: Cannot open audit file"  (this is not a problem)


 #   manuf.model  # display the model parameters

   if (graphics) plot(manuf.model)
   if (graphics) plot(manuf.model, Start=c(1990,1))
   if (graphics) plot(manuf.model, Start=c(1995,1))
   cbps.manuf.model <- bft(trim.na(cbps.manuf.data),verbose=F)
   if (graphics) plot(cbps.manuf.model)
   if (graphics) plot(cbps.manuf.model, Start=c(1995,1))

# to forecast with the model using all available employment data

   z <- forecast(TSmodel(manuf.model), trim.na(manuf.data),   
          conditioning.inputs=trim.na(input.data(manuf.data)))
   if (graphics) plot(z, Start=c(1995,1))

# to see the forecast

   zot <- z$forecast[[1]]
   zot <- window(z$forecast[[1]], start=c(1996,3), warn=F)


  ok <- is.forecast(z)
  if (!ok) {if (is.na(max.error)) max.error <- error
            else max.error <- max(error, max.error)}
  all.ok <- ok 
  if (verbose) 
    {if (ok) cat("ok\n")
     else    cat("failed! (error magnitude= ", error,")\n")
    }

  if (verbose) cat("Guide part 3 test 3 ... ")

   fc <- forecast.cov(manuf.model)
   if (graphics) plot(fc)                                

#relative to zero and trend ...

# to analyse forecast errors

   fc <- forecast.cov(manuf.model,  zero=T, trend=T)
   if (graphics) plot(fc)   

# analysing out-of-sample forecasts errors

#   outfc <-out.of.sample.forecast.cov.estimators.wrt.data(trim.na(manuf.data),
#    estimation.sample=.5,
#    estimation.methods = list(bft=list(verbose=F)), trend=T, zero=T)

   outfc <-out.of.sample.forecast.cov.estimators.wrt.data(trim.na(manuf.data),
    estimation.sample=.5,
    estimation.methods = list(bft=list(verbose=F), est.VARX.ls=NULL),
    trend=T, zero=T)

#   outfc <-out.of.sample.forecast.cov.estimators.wrt.data(
#    trim.na(cbps.manuf.data3),
#    estimation.sample=.5,
#    estimation.methods = list(bft=list(verbose=F), est.VARX.ls=NULL),
#    trend=T, zero=T)

   if (graphics) plot(outfc)



# Sometimes it is useful to send output to a text file rather than to the 
#  screen. To do this use

   sink(file= "zzz.some.name")  #all output goes to file
   ls()
   sink()    # output returns to normal
   unlink("zzz.some.name")

   new.data <- TSdata(manuf.data.ids)  # retrieve new (updated) data set

# To run the model with different data you can use

   z <- l(TSmodel(manuf.model), trim.na(new.data)) 
   z <- l(TSmodel(manuf.model), trim.na(TSdata(manuf.data.ids)))
   if (graphics) plot(z)
   if (graphics) plot(z, Start=c(1995,8))
   z <- forecast(TSmodel(manuf.model), trim.na(new.data),  
     conditioning.inputs=input.data(new.data))

   if (graphics) plot(z, Start=c(1995,6))

#if you actually want the numbers type

   zot <- z$forecast[[1]]

   zot <- window(z$forecast[[1]], start=c(1996,2), warn=F) 
# to put the projected data into a Fame database (at BOC using PADI)

   zot <- output.series.names(z)

   putpadi(z$forecast[[1]],dbname="zzz.nameofdatabase.db", 
        series=output.series.names(z))
   unlink("zzz.nameofdatabase.db")

  ok <- T
  if (!ok) {if (is.na(max.error)) max.error <- error
            else max.error <- max(error, max.error)}
  all.ok <- ok 
  if (verbose) 
    {if (ok) cat("ok\n")
     else    cat("failed! (error magnitude= ", error,")\n")
    }


  if (verbose) cat("Guide part 3 test 4 ... ")

# To give the initial set up before the first time this is run

     manuf.previous.data <- TSdata(manuf.data.ids)
  # then to test (so it looks like the data has changed):
     manuf.previous.data$output[1,1] <- NA

  r <-simple.monitoring(manuf.model, 
          manuf.data.ids, 
          manuf.previous.data,
          mail.list=whoami(),
          message.title="    Manufacturing Monitoring TEST !!!! ",
          message.subject=" TEST !!!! Manufacturing Monitoring",
          show.start= c(0,-3), 
          report.variables=series.names(manuf.data.ids),
          data.sub.heading=
              "   %chg       %chg",
          message.footnote="              f - forecast value" ,
          data.tag=" ",
          forecast.tag="f" )
        # ,save.as=paste("Manufacturing.monitoring.",
        #                 make.names(paste(date(), collapse=".")),sep="")
        # ,run.again=F
  manuf.previous.data <- r[["data"]]
  zot <- r$status

  ok <- T
  if (!ok) {if (is.na(max.error)) max.error <- error
            else max.error <- max(error, max.error)}
  all.ok <- ok 
  if (verbose) 
    {if (ok) cat("ok\n")
     else    cat("failed! (error magnitude= ", error,")\n")
    }


  if (synopsis) 
    {if (verbose) cat("All Brief User Guide example tests part 3 completed")
     if (all.ok) cat(" ok\n")
     else    cat(", some failed! (max. error magnitude= ", max.error,")\n")
    }
  invisible(all.ok)
}


#######################################################################

#                    end

#######################################################################

#   Copyright 1993, 1994, 1995, 1996  Bank of Canada.
#   Copyright 1997 (June), Paul Gilbert.
#   Copyright 1997 (Aug.), Bank of Canada.
#   Copyright 1998, Bank of Canada.

#   The user of this software has the right to use, reproduce and distribute it.
#   Bank of Canada makes no warranties with respect to the software or its 
#   fitness for any particular purpose. The software is distributed by the Bank
#   of Canada and by Paul Gilbert solely on an "as is" basis. By using the  
#   software, user agrees to accept the entire risk of using this software.

################################################################################



############################################################################

#    functions for gradient calculation
#       are in opt.s

############################################################################


project <-function(c1, c2, signif = 0.05, show.details=F, eps=1e-5, fuzz=1e-14)
{
# c1 and c2 should be curvature summary statistics as returned by curve.
# It is assummed that c1 is a submodel of c2.
# The tangent space (parameter effects curvature) of the sub-model c1 is a 
# subspace of the tangent space of the larger model. The acceleration space 
# (first normal space, intrinsic curvature effects) of the sub-model is a 
# subspace of the direct sum of the tangent and acceleration spaces of the 
# larger model, so the intrinsic 
# Tangent and acceleration vectors of the submodel c1 can be projected onto the 
# tangent and acceration spaces of the larger model c2. 
# These are called T1inT2, N1inT2, and N1inN2 (T for tangent, N for normal).
# A second projection (restriction) of c2 onto the tangent and acceleration 
# spaces of c1 is less interesting but may be a useful check. The intrinsic 
# curvature of the larger model should also be intrinsic on the sub-model and
# the parameter effects of the larger model may be partly intrinsic on the 
# sub-model. These two projected onto the intrinsic curvature space of
# the submodel (N2andT2inN1) should compare with the intrinsic curvature of
# the submodel. 

   p1 <- c1$Dlist$p
   N <- length(c1$Dlist$f0)   # dimesion of sample space
   p2 <- c2$Dlist$p
   if (N != length(c2$Dlist$f0)) stop("sample space dimesions do not agree.")
   if (p2 < p1) stop("submodel should be the first argument.")
   if (max(abs(c1$Dlist$f0 - c2$Dlist$f0)) > fuzz)
      warning("projection not computed at the same sample space point as the original curvature calculation.")

   residual <- c(c1$Dlist$f0)   # c() in case it is a column vector
   s.sqr <- sum(residual^2)/(N-2)  # check for time series??

   cat("p1=",p1," p2=",p2,"N=",N,"\n")
   cat("max. span of acceleration space as determined by number of vectors \nin 2nd derivative array:    ")
   cat("   1st model=", p1 *(1+p1)/2,
       ",2nd model=", p2 *(1+p2)/2,"\n" )   

   d1 <- svd(c1$Dlist$D[,1:p1])$d
   d2 <- svd(c2$Dlist$D[,1:p2])$d
   dj <- svd(cbind(c1$Dlist$D[,1:p1], c2$Dlist$D[,1:p2]))$d
   cat("span of tangent vectors:       1st model=", sum(d1 > eps*d1[1]) )
   cat(", 2nd model=",         sum(d2 > eps*d2[1]))
   cat(", jointly=",       sum(dj > eps*dj[1]), "\n")
 
   d1 <- svd(c1$Dlist$D[,-(1:p1)])$d
   d2 <- svd(c2$Dlist$D[,-(1:p2)])$d
   dj <- svd(cbind(c1$Dlist$D[,-(1:p1)], c2$Dlist$D[,-(1:p2)]))$d
   cat("span of acceleration vectors:  1st model=", sum(d1 > eps*d1[1]) )
   cat(", 2nd model=",         sum(d2 > eps*d2[1]))
   cat(", jointly=",       sum(dj > eps*dj[1]), "\n")
   cat("using eps*100 (sensitivity for significance of svd$d )\n")
   cat("span of acceleration vectors:  1st model=", sum(d1 > eps*100*d1[1]) )
   cat(", 2nd model=",         sum(d2 > eps*100*d2[1]))
   cat(", jointly=",       sum(dj > eps*100*dj[1]), "\n")

   m2 <- p2 * (p2 + 3)/2  # m= p+pp is second dimension of D.
   m2 <- min(m2,dim(c2$Dlist$D)[2]) 
   m1 <- p1 * (p1 + 3)/2 

#  c1 projected into c2

   QRofD2 <- qr(c2$Dlist$D)
   T1inT2 <- qr.qty(QRofD2, c1$Dlist$D[,  1:p1 ] )[  1:p2 ,,drop=F]  
   N1inT2 <- qr.qty(QRofD2, c1$Dlist$D[,-(1:p1)] )[  1:p2 ,,drop=F]  
   N1inN2 <- qr.qty(QRofD2, c1$Dlist$D[,-(1:p1)] )[(p2+1):m2,,drop=F]  

#  c2 projected onto c1

   QRofD1 <- qr(c1$Dlist$D)
   N2inN1      <- qr.qty(QRofD1, c2$Dlist$D[,-(1:p2)] )[(p1+1):m1,,drop=F]  
   T2inN1      <- qr.qty(QRofD1, c2$Dlist$D[,  1:p2 ] )[(p1+1):m1,,drop=F] 
   N2andT2inN1 <- qr.qty(QRofD1, c2$Dlist$D           )[(p1+1):m1,,drop=F]
   N2andT2inc1 <- qr.qty(QRofD1, c2$Dlist$D           )

   N1inN1 <- qr.qty(QRofD1, c1$Dlist$D[,(p1+1):m1] )[(p1+1):m1,,drop=F]  
#   v1 <- svd(N2andT2inT1)
#   v2 <- svd(T1inT1)
browser()

   cur2in1 <- rel.curve(s.sqr,N2andT2inc1[1:p2,1:p1], N2andT2inc1[,(p1+1):m1], 
                      show.extra.details=show.details)
   C2in1 <- list(C.parameter=cur2in1[1:p1,,     ,drop=F],
                 C.intrinsic=cur2in1[(p1+1):m1,,,drop=F])

#    z <- rel.curve(s.sqr,N1inN1[1:p1,1:p1], N2andT2inc1[,(p1+1):m1], 
#                       show.extra.details=show.details)
#    zz <- list(C.parameter=z[1:p1,,     ,drop=F],
#                  C.intrinsic=z[(p1+1):m1,,,drop=F])

# svd(   zz$C.intrinsic[1,,])$d
# svd( c1$C$C.intrinsic[1,,])$d
# svd( c2$C$C.intrinsic[1,,])$d
# svd(C2in1$C.intrinsic[1,,])$d

# svd( c1$C$C.intrinsic[1,,])$d / svd(C2in1$C.intrinsic[1,,])$d[1:2]

# eigen( c1$C$C.intrinsic[1,,])$values
# eigen( c2$C$C.intrinsic[1,,])$values
# eigen(C2in1$C.intrinsic[1,,])$values

   effective<-effective.curve(cur1on2,QRofD2, residual, s.sqr,
                      show.details=show.details)

   cstats1on2 <-curve.stats(cur1on2, N, signif=signif)

   R1 <- qr.qty(QRofD1, c2$Dlist$D )[1:m1,,drop=F]  
   cur1 <- rel.curve(s.sqr,R1[1:p1,1:p1], R1[,(p1 + 1):m1], 
                      show.extra.details=show.details)
   effective1<-effective.curve(cur1,QRofD1, residual, s.sqr,
                      show.details=show.details)

   cstats1 <-curve.stats(cur1, N, signif=signif)
browser()

   result <- rbind(c1$stats,c(p2, N, signif, cstats1on2,
                   effective$min.axis.ratio, effective$max.axis.ratio),
                   c2$stats)
   dimnames(result)<-list(c("original c1", "projected", "original c2"),
                          c("P","N","Sign. level","RMS Parameter", 
         "RMS Intrinsic","c*sqrt(F) Parameter","c*sqrt(F) Intrinsic",
         "Min Axis Ratio","Max Axis Ratio")) 
   result
}


#######################################################################

#            calculate Hessian 

#######################################################################



hessian <- function (obj, ...)  UseMethod("hessian")


hessian.default <- function(func, x, func.args=NULL, d=0.01, eps=1e-4,r=6)
{  D <- genD.default(func,x,func.args=func.args, d=d, eps=eps, r=r)$D
   H <- diag(0,length(x))
   u <- 0
   for(i in 1:length(x))
     {for(j in 1:i) 
        {u <- u + 1
         H[i,j] <- D[,u] 
     }  }
   H <- H +t(H)
   diag(H) <- diag(H)/2
   H
}


#######################################################################

#              span (dimension of tangent space) calculation

#######################################################################

span.v00 <- function(func, x,d=0.01, eps=1e-4,r=6, show.details=F){
#  Calculate tangent vectors at the point x. 
#   Note that func can be vector valued.
#
#   This function uses Richardson extrapolation  (for more details      
#      see the functions richardson.grad and genD) to get a numerical 
#      approximation of the tangent vectors to the parameter 
#      manifold. SVD is then used to calculate their span.     
#
#       func    the name of a function which returns the
#                residual vector for a given parameter vector.
#       x       The parameters of func.
#
#       d       initial interval value by
#               default set to 0.01*x or eps if x is 0.0 .
#       r       the number of repetions with successly smaller d. 
#       show.details    if TRUE then detailed calculations are shown.
#       v       reduction factor. This could be a parameter but ...
#               N.B. must be 2 for richarson formula as used.

  v <- 2
  p <- length(x)     #  number of parameters (= dim of M if not over parameterized.
  n <- len(func(x))  #  dimension of sample space.
  D <- array(1, c(n, p, r)) 
  h <- abs(d*x)+eps*(x==0.0)
  for(k in 1:r)  # successively reduce h 
    {for(i in 1:p)  # each parameter  - first deriv.
       D[,i,k]<-(func(x+(i==(1:p))*h)-func(x-(i==(1:p))*h))/(2*h[i]) # F'(i)
     h <- h/v     # Reduced h by 1/v.
    }	
  if(show.details)  
      {cat("\n","first order approximations", "\n")		
       for(i in 1:p){ cat(" parameter " ,i,"\n");print(D[,i,(1:r)], 12)}
      }
  for(m in 1:(r - 1)) 		
     {D<- (D[,,2:(r+1-m)]*(4^m)-D[,,1:(r-m)])/(4^m-1)
      if(show.details & m!=(r-1) )  
        {cat("\n","Richarson improvement group No. ", m, "\n")		
         for(i in 1:p){ cat(" parameter " ,i,"\n"); print(D[,i,(1:(r-m))], 12) }
        }
      }
   svd(D)$d
}


span.v0 <- function(func, x,d=0.01, eps=1e-4,r=6, show.details=F){
#  span performs a svd of the tangent vectors at the point x. 
#   (Note that func can be vector valued. This can be used
#   to calculate the dimension of the tangent space (ie. by over specifying 
#   the model and counting the number of significant singular values).
#  The singular values are returned.
#
#   This function uses Richardson extrapolation  (for more details      
#      see the functions richardson.grad and genD) to get a numerical 
#      approximation of the tangent vectors to the parameter 
#      manifold. SVD is then used to calculate their span.     
#
#       func    the name of a function which returns the
#                residual vector for a given parameter vector.
#       x       The parameters of func.
#
#       d       initial interval value by
#               default set to 0.01*x or eps if x is 0.0 .
#       r       the number of repetions with successly smaller d. 
#       show.details    if TRUE then detailed calculations are shown.
#       v       reduction factor. This could be a parameter but ...
#               N.B. must be 2 for richarson formula as used.

  v <- 2
  p <- length(x)     #  number of parameters (= dim of M if not over parameterized.
  n <- len(func(x))  #  dimension of sample space.
  D <- array(1, c(n, p, r)) 
  h <- abs(d*x)+eps*(x==0.0)
  for(k in 1:r)  # successively reduce h 
    {for(i in 1:p)  # each parameter  - first deriv.
       D[,i,k]<-(func(x+(i==(1:p))*h)-func(x-(i==(1:p))*h))/(2*h[i]) # F'(i)
     h <- h/v     # Reduced h by 1/v.
    }	
  if(show.details)  
      {cat("\n","first order approximations", "\n")		
       for(i in 1:p){ cat(" parameter " ,i,"\n");print(D[,i,(1:r)], 12)}
      }
  for(m in 1:(r - 1)) 		
     {D<- (D[,,2:(r+1-m)]*(4^m)-D[,,1:(r-m)])/(4^m-1)
      if(show.details & m!=(r-1) )  
        {cat("\n","Richarson improvement group No. ", m, "\n")		
         for(i in 1:p){ cat(" parameter " ,i,"\n"); print(D[,i,(1:(r-m))], 12) }
        }
      }
   svd(D)$d
}

span <- function (obj, ...)  UseMethod("span")

span.default <- function(func, x, func.args=NULL, d=0.01, eps=1e-4,r=6, show.details=F)
{
#   performs a svd of the tangent vectors at the point x. This can be used
#   to calculate the dimension of the tangent space (ie. by over specifying 
#   the model and counting the number of significant singular values).
#  The singular values are returned.
#
#   This function uses Richardson extrapolation  (for more details      
#      see the functions richardson.grad and genD) to get a numerical 
#      approximation of the tangent vectors to the parameter 
#      manifold. SVD is then used to calculate their span.     
#
#  func     a charater string of the name of a function which returns the
#                residual vector for a given parameter vector.
#   x       The parameters of func with respect to which derivative is calculated.
#  func.args A list of any other arguments to the function
#
#   d       initial interval value by
#               default set to 0.01*x or eps if x is 0.0 .
#   r       the number of repetions with successly smaller d. 
#   show.details    if TRUE then detailed calculations are shown.
#   v       reduction factor. This could be a parameter but ...
#             N.B. must be 2 for richarson formula as used.

  v <- 2
  p <- length(x)     #  number of parameters (= dim of M if not over parameterized.
  n <- length(do.call(func,append(list(x), func.args)))  #dimension of sample space.
  D <- array(1, c(n, p, r)) 
  h <- abs(d*x)+eps*(x==0.0)
  for(k in 1:r)  # successively reduce h 
    {if(show.details) cat("\n k=",k, " p: ")
     for(i in 1:p)  # each parameter  - first deriv.
      {if(show.details) cat(" ",i)
       D[,i,k]<-(do.call(func,append(list(x+(i==(1:p))*h), func.args)) -
                 do.call(func,append(list(x-(i==(1:p))*h), func.args)))/(2*h[i])
      }
     h <- h/v     # Reduced h by 1/v.
    }	
  if(show.details)  
      {cat("\n","first order approximations", "\n")		
       for(i in 1:p){ cat(" parameter " ,i,"\n");print(D[,i,(1:r)], 12)}
      }
  for(m in 1:(r - 1)) 		
     {D<- (D[,,2:(r+1-m)]*(4^m)-D[,,1:(r-m)])/(4^m-1)
      if(show.details & m!=(r-1) )  
        {cat("\n","Richarson improvement group No. ", m, "\n")		
         for(i in 1:p){ cat(" parameter " ,i,"\n"); print(D[,i,(1:(r-m))], 12) }
        }
      }
   svd(D)$d
}


#######################################################################

#                            curvature calculation

#######################################################################


print.curvature <- function (obj, ...)  { print(obj$stats, ...) }

curve <- function (obj, ...)  
{ # calculate the Bates and Watts intrinsic and parameter effects curvature.
  UseMethod("curve")
}


curve.default <- function(func, x, func.args=NULL, d=0.01, eps=1e-4,r=6, show.details=F)
{D <-genD.default(func,x, func.args=func.args, d=d, eps=eps,r=r)
 curve(D)
}


curve.Darray <-function(Dlist, signif = 0.05, show.extra.details=F, show.details=show.extra.details)
{
# Curvature summary statistics as in Bates and Watts.
#  Dlist is a list as generated by genD.default and genD.TSestModel,
#       with the 3 elements as follows:
#   $D is a matrix of first(gradients) and second order partial
#      derivatives organized in the same manner as Bates and 
#      Watts. (first p columns are the gradients and the 
#      next p(p+1)/2 columns are the lower triangle of the Hessian).
#   $p is the dimension of the parameter space=dim of the tangent space.
#   $f0 is the function value at the point where the matrix D 
#        was calculated. (The curvature calculation should not/does not? depend 
#        on this value - but it should be the right dimension and 0's do
#        not work.

#  Page references are to Bates and Watts(1983), Nonlinear Regression 
#   Analysis and Its Applications.
#   Examples are also from this text. e.g.- model A data set 3.

#   modified substantially by Paul Gilbert (May, 1992 and Dec, 1996)
#    from original code by Xingqiao Liu,   May, 1991.
#
   if (all(Dlist$D==0))
     stop("The array of first and second order derivatives is all zeros.")

   p <- Dlist$p
   n <- dim(Dlist$D)[1]   # dimesion of sample space
   pp <- p*(p+1)/2 # p' max number of independent col in the acceleration array.
   m <- p * (p + 3)/2  # m= p+pp is second dimension of D. It is also the max 
                       # span of the combined tangent and accelation spaces.
   if (m!=dim(Dlist$D)[2]) stop("Dimesion of D matrix is not consistent with p")
   residual <- c(Dlist$f0)   # c() in case it is a column vector
   s.sqr <- sum(residual^2)/(length(residual)-2)
   if(show.extra.details) cat("p=",p," pp=",pp," m=",m, " n=",n,"\n")
 
#  Form QR decomposition of D and produce matrix Q and R.
#

   QRofD <- qr(Dlist$D) 

# Now calculate R, which is D rotated onto an othogonal basis with the 
# tangent space first, then acceleration space. 
# In this basis first rows of R are in the tangent and the next rows are in the
# acceleration space. Following rows would be
# all zeros  and so those rows are truncated from R.

#   Q <- qr.qy(QRofD, diag(1,n)) [,1:m]
#   R <- t(Q) %*% Dlist$D   this is the same as
#  R[1:m,]could be used to truncate but it is possible for m to exceed dim(D)[1]
   R <- qr.qty(QRofD, Dlist$D )  

   if(show.extra.details) 
        {cat( "Matrix D, tangent and acceleration vectors in columns ")
         cat("organized as in Bates and Watts p235, 236\n")
         print(Dlist$D)
         cat("Residual Vector\n"); print(residual)
	 cat("Matrix Q from QR decomposition of D\n")
         print( qr.qy(QRofD, diag(n)))
	}
   if(show.details) 
        {cat("Matrix R (organized as in Bates and Watts R1 p236)\n")
         print(R)
	}

#  The matrix R partitions into p by p matrix R11 and m by p'  matrix R2.
#  R2 is Bates and Watts  R12 
#                         R22  p236

#   R11 <- R[1:p,1:p]
#   R2 <- R[1:m,(p + 1):m] 

#                                     R[1:m,(p + 1):m]
   cur <- rel.curve(s.sqr,R[1:p,1:p], R[,(p + 1):m], 
                      show.extra.details=show.extra.details)

   if (m>dim(cur)[1]) 
     {m <- dim(cur)[1]
      warning("acceration space dimension reduced for smaller sample space.")
     }
   C <- list(C.parameter=cur[1:p,,    ,drop=F],
             C.intrinsic=cur[(p+1):m,,,drop=F])
   class(C) <-"curvatureArray"

# effective residual curvature. Bates and Watts p260
# Calculate the scaled RMS curvatures relative to a confidence disk 
# radius and extreme axis ratios. ref. Bates and Watts p254 eqn (7.23).
#  and Bates and Watts J.R. Statist.Soc. B (1980).

   effective<-effective.curve(cur,QRofD, residual, s.sqr, 
                              show.details=show.extra.details)
   cstats <-curve.stats(cur, n, signif=signif)

   result <- matrix(c(p, n, signif, cstats,
               effective$min.axis.ratio, effective$max.axis.ratio),1,9)
   dimnames(result)<-list(NULL,c("P","N","Sign. level","RMS Parameter", 
         "RMS Intrinsic","c*sqrt(F) Parameter","c*sqrt(F) Intrinsic",
         "Min Axis Ratio","Max Axis Ratio")) 
   result <-list(stats=result, Dlist=Dlist, C=C)
   class(result) <- "curvature"
   result
}

print.curvatureArray <- function(x, ...)
  {cat("Parameter effects relative curvature array:\n")
   for (i in 1:dim(x$C.parameter)[1]) 
       {print(x$C.parameter[i,,], ...); cat("\n")}
   cat("Intrinsic relative curvature array:\n")
   for (i in 1:dim(x$C.intrinsic)[1])
       {print(x$C.intrinsic[i,,], ...); cat("\n")}
   invisible(x)
  }

curve.stats <-function(cur, n, signif=0.05)
{# See documentation in in curve.Darray
 p  <- dim(cur)[2]
 pp <- dim(cur)[1] - p
 cstats <- rep(NA,4)
   #  i from 1 to p for the RMS parameter effects curvature.
   #  i from p+1 to m=p+p' for the RMS intrinsic effects curvature.
   di <- 0
   for(i in 1:p) di <- di + sum(diag(cur[i,,]))^2
   cstats[1] <-  2*sum(cur[1:p,,]^2) + di
   cstats[1] <-  sqrt(cstats[1]/(p*(p+2)))
   di <- 0
   for(i in p+(1:pp)) di <- di + sum(diag(cur[i,,]))^2
   cstats[2] <-  2*sum(cur[p+(1:pp),,]^2) + di
   cstats[2] <-  sqrt(cstats[2]/(p*(p+2)))
  value.f <- qf(1 - signif, p, n - p)	# F(p, n-p; signif)
  cstats[3] <- cstats[1] * sqrt(value.f) # c*sqrt(F) Parameter
  cstats[4] <- cstats[2] * sqrt(value.f) # c*sqrt(F) Intrinsic
  cstats
}


effective.curve <- function(cur,QRofD, residual, s.sqr, show.details=F)
{# Transform the residual vector by multiply Q-transpose and sqrt(s.sqr*p).
 # Calculate the p by p effective residual curvature matrix B and its 
 #  eigenvalues. Bates and Watts p260
 p  <- dim(cur)[2]
 pp <- dim(cur)[1] - p
 Qz <- qr.qty(QRofD, residual)[p+(1:pp)]/c(sqrt(p * s.sqr))  # p' vector of rotated residuals
 #pxp effective residual (intrinsic) curvature matrix. Bates and Watts p260
   # Qz is a vector of length pp.  
   B <- matrix(NA,p,p) 
   A.intrin <- cur[p+(1:pp),,]
 #   for (i in 1:p)
 #       for (j in 1:p)
 #         B[i,j] <-  A.intrin[,i,j] %*% Qz 
 # following is equivalent to above
 B <- apply(A.intrin * Qz, c(2,3), sum)
 if(show.details) 
     { cat("Scaled and Q-transformed Residual Vector\n"); print(Qz)
       cat("Effective residual curvature matrix B (Bates and Watts p 260)\n")
       print(B)
     }
 if ( max(abs(B- t(B))) > 1e-7 ) warning("B is not symmetric.")  #Rbug
   eigv <- eigen(B)$values

   if(show.details)   {cat("Eigenvalues of B\n"); print(eigv )}

   if(max(eigv) >= 1.0)
     {warning( paste(
      "N.B. (I-B) is not positive definite as it should be at a local min! ",
      "Axis ratio will evaluate to NA. ",
      "Eigenvalues of B ", eigv))
     }

 #Rbug Mod in the following might be better, but the eigv should be real
# axis.ratios <- 1/sqrt(1-eigv)
# list(B=B, eigenvalues=eigv,
#      min.axis.ratio=min(axis.ratios),
#      max.axis.ratio=max(axis.ratios)  )
# Bates and Watts seem to use the following but the above should be considered.
# It seems possible to have an eigv < -1
 list(B=B, eigenvalues=eigv,
      min.axis.ratio=1/sqrt(1-min(abs(eigv))),
      max.axis.ratio=1/sqrt(1-max(abs(eigv)))  )
}

rel.curve <-function(s.sqr,R11,R2, show.extra.details=F, 
                     eps=sqrt(.Machine$double.eps))
{# The result C is the relative curvature array Bates & Watts p244 eqn. (7.16)
 # R11 is a p by p sub matrix R see Bates & Watts p236
 #  and  R2 is the m by pp sub matrix  R12 
 #                                     R22   

 p  <- dim(R11)[2]
 p2 <- dim(R11)[1]  # usually equal p except for projections
 pp <- dim(R2)[2]
 # m will usually be p+pp, but if the number of observations is small relative 
 # to pp, then Q and R2 will be truncated.
 # For smaller problems it would be quicker to do the calculation just to p+pp
 # but small problems are quick anyway.
 m  <- dim(R2)[1] 
 A  <- array(NA,c(m,p,p))  # for accel. array B&W (7.5)
 C  <- array(NA,c(m,p2,p2))
   
# R11.inv <- solve(R11)
 v <-svd(R11)
 if (all(v$d == 0)) stop("R11 is completely degenerate.")
 d <- 1/v$d
 illim <- v$d < (v$d[1] * eps)
 if (any(illim)) 
   {warning("eliminating degenerate subspace for R11.")
    d[illim] <- 0
   }
 R11.inv <- v$v %*% diag(d) %*% t(v$u) 

 R11.invT <- t(R11.inv)
 if(show.extra.details) 
      { cat("Bates and Watts matrix R11 p236)\n")
        print(R11) 
        cat("Matrix R2 (Bates and Watts  R12 \n")
        cat("                            R22 )  p236)\n")
        print(R2)
        cat("Inverse Of Matrix R11  e.g. Bates and Watts p243\n")
        print(R11.inv)
      }
  # expand R2 to make A
  # there may be a better way to do this without expanding
   k <- 0
   for(i in 1:p) 
     for(j in 1:i) 
       {k <- k + 1
        A[,i,j] <- R2[, k]
        A[,j,i] <- R2[, k]
       }
   if(show.extra.details)
     {cat("Acceleration array (Bates and Watts matrix A p237).\n")
      cat("Parameter effects relative acceleration array:\n")
      for (i in 1:p) {print(A[i,,]); cat("\n")}
      cat("Intrinsic relative acceleration array:\n")
      for (i in (p+1):m) {print(A[i,,]); cat("\n")}
     }	

# compute C  Batts and Watts defn. (7.16) p242 and examples p244 &  p245
   for(i in 1:m)
      C[i,,]<-(R11.invT %*% A[i,,] %*% R11.inv) * c(sqrt(p * s.sqr))   

if(show.extra.details)
  {cat("Relative curvature array (Bates and Watts matrix C p242 and eg p 243- 245).\n")
      cat("Parameter affects curvature array:\n")
      for (i in 1:p) {print(C[i,,]); cat("\n")}
      cat("Intrinsic curvature array:\n")
      for (i in (p+1):m) {print(C[i,,]); cat("\n")}
  }	

C
}


#######################################################################

#                               D matrix calculation

#######################################################################

# Notes:

#   Generating the D matrix can be computationally demanding. There are two
#   versions of the code for generating this matrix. The S version is 
#   slow and suffers from memory problems due to the way S allocates memory 
#   in loops (as of S-PLUS Version 3.0 for SUN4). The C version is
#   faster but suffers (even worse) from the memory problem.  There is a third
#   version which is written in S but using C code logic (for loops). This 
#   version is primarily for debugging the C code. Both the S and 
#   the C versions take the name of an S function 
#   as an arguement (and call it multiple times).  
#######################################################################
genD <- function(arg,...)UseMethod("genD")

genD.default <- function(func, x, d=0.01, eps=1e-4, r=6, func.args=NULL){
# This function calculates a numerical approximation of the first and second
#   derivatives (gradient and hessian) of func at the point x. The calculation
#   is done by Richardson's extrapolation (see eg. G.R.Linfield and J.E.T.Penny
#   "Microcomputers in Numerical Analysis"). The method should be used when
#   accuracy, as opposed to speed, is important.
# The result returned is a list with 3 elements as follows:
#   $D is a matrix of first(gradients) and second order partial
#      derivatives organized in the same manner as Bates and 
#      Watts. (first p columns are the gradients and the 
#      next p(p-1)/2 columns are the lower triangle of the Hessian).
#   $p is the dimension of the parameter space=dim of the tangent space.
#   $f0 is the function value at the point where the matrix D 
#        was calculated. 
#   Also the arguments (func, x, d, eps, r) are returned
#
#   modified substantially by Paul Gilbert (May, 1992)
#    from original code by Xingqiao Liu,   May, 1991.
#

#  func    name of the function. The first arg to func must be a vector.
#  x       the parameter vector.
#  func.args any additional arguments to func.
#  d       gives the fraction of x to use for the initial numerical approximation.
#  eps     is used for zero elements of x.
#  r       the number of Richardson improvement iterations.
#  v      reduction factor for Richardson iterations. This could
#      be a parameter but the way the formula is coded it is assumed to be 2.

#  Modified substantially by Paul Gilbert from first version by Xingqiao Liu.
#  This function is not optimized for S speed, but
#  is organized in the same way it is implemented in C (to facilitate checking
#  the code).

#         - THE FIRST ORDER DERIVATIVE WITH RESPECT TO Xi IS 
#           F'(Xi)={F(X1,...,Xi+d,...,Xn) - F(X1,...,Xi-d,...,Xn)}/(2*d)
#                
#         - THE SECOND ORDER DERIVATIVE WITH RESPECT TO Xi IS 
#           F''(Xi)={F(X1,...,Xi+d,...,Xn) - 2*F(X1,X2,...,Xn)+
#                     F(X1,...,Xi-d,...,Xn)}/(d^2)
#
#         - THE SECOND ORDER DERIVATIVE WITH RESPECT TO Xi, Xj IS 
#           F''(Xi,Xj)={F(X1,...,Xi+d,...,Xj+d,...,Xn)-2*F(X1,X2,...,Xn)+
#                       F(X1,...,Xi-d,...,Xj-d,...,Xn)}/(2*d^2) -
#                      {F''(Xi) + F''(Xj)}/2

#  The size of d is iteratively reduced and the Richardson algorithm is applied to
#    improve the accuracy of the computation.

   v <-2
   zt <- .001     #Rbug unix.time has trouble finding func
   # f0 <- func(x)
   f0 <- do.call(func,append(list(x), func.args))
 # zt <-unix.time(f0 <- func(x))   #  f0 is the value of the function at x.
                   # (Real value or point in sample space ie- residual vector)	
   p <- length(x)  #  number of parameters (theta)
   zt <- zt[1] *r*2*(p*(p + 3))/2
   if(zt>500) 
     cat("D matrix calculation roughly estimated to take about ",
          (10*ceiling(zt/10)),"seconds(without other system activity)...\n")
   h0 <- abs(d*x)+eps*(x==0.0)
   D <- matrix(0, length(f0),(p*(p + 3))/2)
   #length(f0) is the dim of the sample space
   #(p*(p + 3))/2 is the number of columns of matrix D.( first
   #   der. & lower triangle of Hessian)
   Daprox <- matrix(0, length(f0),r) 
   Hdiag  <-  matrix(0,length(f0),p)
   Haprox <-  matrix(0,length(f0),r)
   for(i in 1:p)    # each parameter  - first deriv. & hessian diagonal
        {h <-h0
         for(k in 1:r)  # successively reduce h 
           {#f1 <- func(x+(i==(1:p))*h)
            #f2 <- func(x-(i==(1:p))*h) 
            f1 <- do.call(func,append(list(x+(i==(1:p))*h), func.args))
            f2 <- do.call(func,append(list(x-(i==(1:p))*h), func.args))
            Daprox[,k] <- (f1 - f2)  / (2*h[i])    # F'(i) 
            Haprox[,k] <- (f1-2*f0+f2)/ h[i]^2     # F''(i,i) hessian diagonal
            h <- h/v     # Reduced h by 1/v.
           }
         for(m in 1:(r - 1))
            for ( k in 1:(r-m))
              {Daprox[,k]<-(Daprox[,k+1]*(4^m)-Daprox[,k])/(4^m-1)
               Haprox[,k]<-(Haprox[,k+1]*(4^m)-Haprox[,k])/(4^m-1)
              }
         D[,i] <- Daprox[,1]
         Hdiag[,i] <- Haprox[,1]
        }	
   u <- p

   for(i in 1:p)   # 2nd derivative  - do lower half of hessian only
     {for(j in 1:i) 
        {u <- u + 1
            if (i==j) { D[,u] <- Hdiag[,i] }
            else 
             {h <-h0
              for(k in 1:r)  # successively reduce h 
                {#f1 <- func(x+(i==(1:p))*h + (j==(1:p))*h)
                 #f2 <- func(x-(i==(1:p))*h - (j==(1:p))*h)
                 f1 <- do.call(func, append(
                        list(x+(i==(1:p))*h + (j==(1:p))*h), func.args))
                 f2 <- do.call(func,append(
                        list(x-(i==(1:p))*h - (j==(1:p))*h), func.args))  
                 Daprox[,k]<- (f1 - 2*f0 + f2 -
                                  Hdiag[,i]*h[i]^2 - 
                                  Hdiag[,j]*h[j]^2)/(2*h[i]*h[j])  # F''(i,j)  
                 h <- h/v     # Reduced h by 1/v.
                }
              for(m in 1:(r - 1))
                 for ( k in 1:(r-m))
                   Daprox[,k]<-(Daprox[,k+1]*(4^m)-Daprox[,k])/(4^m-1)
              D[,u] <- Daprox[,1]
             }
          }  
       }
D <- list(D=D,p=length(x),f0=f0, func=func, x=x, d=d, eps=eps, r=r)
class(D) <- "Darray"
invisible(D)
}

genD.default.v2 <- function(func, x, func.args=NULL, d=0.01, eps=1e-4, r=6){
# This function calculates a numerical approximation of the first and second
#   derivatives (gradient and hessian) of func at the point x. The calculation
#   is done by Richardson's extrapolation (see eg. G.R.Linfield and J.E.T.Penny
#   "Microcomputers in Numerical Analysis"). The method should be used 
#   accuracy, as opposed to speed, is important.
# The result returned is a list with 3 elements as follows:
#   $D is a matrix of first(gradients) and second order partial
#      derivatives organized in the same manner as Bates and 
#      Watts. (first p columns are the gradients and the 
#      next p(p-1)/2 columns are the lower triangle of the Hessian).
#   $p is the dimension of the parameter space=dim of the tangent space.
#   $f0 is the function value at the point where the matrix D 
#        was calculated. 
#
#   modified substantially by Paul Gilbert (May, 1992)
#    from original code by Xingqiao Liu,   May, 1991.
#
#  This function is not optimized for S speed, but
#  is organized in the same way it is implemented in C (to facilitate  
#  checking the code).


#  func    char string name of the function. func must have a single vector arguement.
#  x       the parameter vector.
#  d       gives the fraction of x to use for the initial numerical approximation.
#  eps     is used for zero elements of x.
#  r       the number of Richardson improvement iterations.
#  v      reduction factor for Richardson iterations. This could
#      be a parameter but the way the formula is coded it is assumed to be 2.

#         - THE FIRST ORDER DERIVATIVE WITH RESPECT TO Xi IS 
#           F'(Xi)={F(X1,...,Xi+d,...,Xn) - F(X1,...,Xi-d,...,Xn)}/(2*d)
#                
#         - THE SECOND ORDER DERIVATIVE WITH RESPECT TO Xi IS 
#           F''(Xi)={F(X1,...,Xi+d,...,Xn) - 2*F(X1,X2,...,Xn)+
#                     F(X1,...,Xi-d,...,Xn)}/(d^2)
#
#         - THE SECOND ORDER DERIVATIVE WITH RESPECT TO Xi, Xj IS 
#           F''(Xi,Xj)={F(X1,...,Xi+d,...,Xj+d,...,Xn)-2*F(X1,X2,...,Xn)+
#                       F(X1,...,Xi-d,...,Xj-d,...,Xn)}/(2*d^2) -
#                      {F''(Xi) + F''(Xj)}/2

#  The size of d is iteratively reduced and the Richardson algorithm is applied to
#    improve the accuracy of the computation.

   v <-2
   f0 <- do.call(func, append(list(x), func.args))  #  f0 is the value of the function at x.
   p <- length(x)  #  number of parameters (theta)
   h0 <- abs(d*x)+eps*(x==0.0)
   D <- matrix(0, length(f0),(p*(p + 3))/2)
   #length(f0) is the dim of the sample space
   #(p*(p + 3))/2 is the number of columns of matrix D.( first
   #   der. & lower triangle of Hessian)
   Daprox <- matrix(0, length(f0),r) 
   Hdiag  <-  matrix(0,length(f0),p)
   Haprox <-  matrix(0,length(f0),r)
   for(i in 1:p)    # each parameter  - first deriv. & hessian diagonal
        {h <-h0
         for(k in 1:r)  # successively reduce h 
           {f1 <- do.call(func, append(list(x+(i==(1:p))*h), func.args))
            f2 <- do.call(func, append(list(x-(i==(1:p))*h), func.args))
            Daprox[,k] <- (f1 - f2)  / (2*h[i])    # F'(i) 
            Haprox[,k] <- (f1-2*f0+f2)/ h[i]^2     # F''(i,i) hessian diagonal
            h <- h/v     # Reduced h by 1/v.
           }
         for(m in 1:(r - 1))
            for ( k in 1:(r-m))
              {Daprox[,k]<-(Daprox[,k+1]*(4^m)-Daprox[,k])/(4^m-1)
               Haprox[,k]<-(Haprox[,k+1]*(4^m)-Haprox[,k])/(4^m-1)
              }
         D[,i] <- Daprox[,1]
         Hdiag[,i] <- Haprox[,1]
        }	
   u <- p

   for(i in 1:p)   # 2nd derivative  - do lower half of hessian only
     {for(j in 1:i) 
        {u <- u + 1
            if (i==j) { D[,u] <- Hdiag[,i] }
            else 
             {h <-h0
              for(k in 1:r)  # successively reduce h 
                {f1 <- do.call(func, append(list(x+(i==(1:p))*h + (j==(1:p))*h), func.args))
                 f2 <- do.call(func, append(list(x-(i==(1:p))*h - (j==(1:p))*h), func.args))  
                 Daprox[,k]<-(f1-2*f0+f2-Hdiag[,i]*h[i]^2-Hdiag[,j]*h[j]^2)/(2*h[i]*h[j])  # F''(i,j)  
                 h <- h/v     # Reduced h by 1/v.
                }
              for(m in 1:(r - 1))
                 for ( k in 1:(r-m))
                   Daprox[,k]<-(Daprox[,k+1]*(4^m)-Daprox[,k])/(4^m-1)
              D[,u] <- Daprox[,1]
             }
          }  
       }
D <- list(D=D,p=length(x),f0=f0)
class(D) <- "Darray"
invisible(D)
}

genD.c <- function(func, x, d=0.01, eps=1e-4, r=6){
# See documentation in genD and genD.default 
#evalNo <<-0
#cat("evalNo = ")
   v <-2
   zt <- .001     #Rbug unix.time has trouble finding func
   f0 <- func(x)
#   zt<- unix.time(f0 <- func(x) )
   h0 <- abs(d*x)+eps*(x==0.0)
   p <- length(x)
   zt <- zt[1] *r*2*(p*(p + 3))/2
   if(zt>30) 
     cat("D matrix calculation roughly estimated to take about ",
         (10*ceiling(zt/10)),"seconds(without other system activity)...\n")
   D <- matrix(0, length(f0),(p*(p + 3))/2) 
   Daprox <-  matrix(0,length(f0),r) 
   Hdiag  <-  matrix(0,length(f0),p)
   Haprox <-  matrix(0,length(f0),r)
   storage.mode(D)     <-"double"
   storage.mode(Daprox)<-"double"
   storage.mode(Haprox)<-"double"
   storage.mode(Hdiag )<-"double"
   D<-.C("gen_D",
            list(func),             #1
            p=as.integer(length(x)),  #2
            as.double(x),
            as.double(h0),
            as.integer(length(f0)), #5
            f0=as.double(f0),       #6
            as.integer(r),
            as.integer(v),
            Haprox,        #9
            Hdiag,         #10
            Daprox,        #11
            double(length(x)),
            double(length(f0)),
            double(length(f0)),
            D=D)[c("D","p","f0")]
   class(D) <- "Darray"
   invisible(D)
}


load.curve.c    <- function ()
{# load C routines for use by S functions genD.c (which is NOT an improvement
 # over the S version) and R11T.c (which doesn't work).
 dyn.load(paste(DSE.HOME,"/curve.o", sep=""))
} 

R11T.c <- function(R11.inv,p,pp){
# See documentation in R11T.s (in curve) 
warning("usage of pp in R11T.c does not seem to be right.")
cat("this version does not work!")
   storage.mode(R11.inv)  <-"double"
   .C("R11T_c",
            R11T=matrix(0,pp,pp),
            R11.inv,
            as.integer(p),
            as.integer(pp))[["R11T"]]
}


#######################################################################

# Test routines and data for calulating curvatures in Bates and Watts.

#######################################################################
 
curve.function.tests <- function( verbose=T, synopsis=T, fuzz.small=1e-14, fuzz.large=1e-6, show.details=F, show.extra.details=F)
{# A short set of tests of curvature functions

  max.error <- 0
  if (synopsis & !verbose) cat("All curvature tests ...")

  if (verbose) cat("curvature test 1 ... ")


signif <- 0.05

#Rbug do.call in genD has trouble with local functions
global.assign("puromycin", function(th)
  {x <- c(0.02,0.02,0.06,0.06,0.11,0.11,0.22,0.22,0.56,0.56,1.10,1.10)
   y <- c(76,47,97,107,123,139,159,152,191,201,207,200)
   ( (th[1] * x)/(th[2] + x) ) - y
  })
global.assign("D.anal", function(th)
 {# analytic derivatives. Note numerical approximation gives a very good
  # estimate of these, but neither give D below exactly. The results are very
  # sensitive to th, so rounding error in the reported value of th could explain
  # the difference. But more likely th is correct and D has been rounded for
  # publication - and the analytic D with published th seems to work best.
  # th = c(212.70188549 ,  0.06410027) is the nls est of th for BW published D.
  x <- c(0.02,0.02,0.06,0.06,0.11,0.11,0.22,0.22,0.56,0.56,1.10,1.10)
  y <- c(76,47,97,107,123,139,159,152,191,201,207,200)
  cbind(x/(th[2]+x), -th[1]*x/(th[2]+x)^2, 0, -x/(th[2]+x)^2, 2*th[1]*x/(th[2]+x)^3)
 })
on.exit(rm(puromycin, D.anal))

# D matrix from p235. This may be useful for rough comparisons, but rounding
# used for publication introduces substanstial errors. check D.anal1-D.BW
D.BW <- t(matrix(c(
0.237812, -601.458, 0, -2.82773, 14303.4,
0.237812, -601.458, 0, -2.82773, 14303.4,
0.483481, -828.658, 0, -3.89590, 13354.7,
0.483481, -828.658, 0, -3.89590, 13354.7,
0.631821, -771.903, 0, -3.62907, 8867.4,
0.631821, -771.903, 0, -3.62907, 8867.4,
0.774375, -579.759, 0, -2.72571, 4081.4,
0.774375, -579.759, 0, -2.72571, 4081.4,
0.897292, -305.807, 0, -1.43774,  980.0,
0.897292, -305.807, 0, -1.43774,  980.0,
0.944936, -172.655, 0, -0.81173,  296.6,
0.944936, -172.655, 0, -0.81173,  296.6),   5,12))

   D.anal <- D.anal(c(212.7000, 0.0641))
#   D.anal2 <- D.anal(c(212.683, 0.0641194))
   D.calc <- genD.default("puromycin",c(212.7000, 0.0641))
#  D.calc2 <- genD.default(puromycin,c(212.683, 0.0641194))$D #est using nls
#check if col 4 is a mult. of col 2 max(abs(D.anal1-D.calc1$D))

   if (show.details)
     {cat("model A p329,data set 3 (table A1.3, p269) Bates & Watts (Puromycin example).\n")
      cat("With show.extra.details=T the calculations  on pages 235 - 252 are shown.\n")
      cat("Note Bates & Watts calculation uses s squared = 10.93^2, which is\n")
      cat(" based on both treated and untreat data. The treated data alone\n")
      cat("gives 10.39^2. This causes a slight difference in the curvature statistics.\n") 
      cat("Note that the R22 calculation using numerical gradients is quite\n")
      cat("different from the published result, but still gives similar results.\n")
      }

   r.BW   <- c(2, 12, 0.05, 0.105,        0.045,      0.21,       0.09,        1, 1.05)
   r.test <- c(2, 12, 0.05, 0.1046988133, 0.04541991, 0.21207185, 0.091999935, 1, 1.051959666)
   r.calc <-curve.Darray(D.calc,signif=signif, 
                        show.extra.details=show.details)$stats 
   err <- r.calc - r.test
   max.error <- max(max.error, abs(err))
   ok <- all(abs(err) < fuzz.large)
   if (show.details | !ok) 
     {cat("Using numerical calculation of gradient and acceration from data:\n")
      m <- rbind(r.BW, r.test, r.calc, err)
      dimnames(m) <- list(
                    c("Bates&Watts","test value","calculated ", "difference "),
                    c(dimnames(r.calc)[[2]]))
      print(m)
      cat("Above should compared with Bates and Watts table 7.2, p 257, model A, data set 3.\n")
     }
   r.calc <- curve.Darray(list(p=2,D=D.anal,f0=puromycin(c(212.7000, 0.0641))),
                    signif=signif, show.extra.details=show.details)$stats 
   err <- r.calc - r.test
   ok2 <- all(abs(err) < fuzz.large)
   max.error <- max(max.error, abs(err))
   if (show.details | !ok2) 
     {cat("Using D matrix of gradient and acceration calcuated with formulae Bates and Watts p234:\n")
      m <- rbind(r.BW, r.test, r.calc, err)
      dimnames(m) <- list(
                    c("Bates&Watts","test value","calculated ", "difference "),
                    c(dimnames(r.calc)[[2]]))
      print(m)
     }
   ok3 <- max(abs(D.calc$D-D.anal)) < (100 * fuzz.large)
   if (!ok3)
     {cat("max diff. between analtic and numerical D:")
      cat( max(abs(D.calc$D-D.anal)), "\n")
     }
  ok <- ok & ok2 & ok3
  all.ok <- ok
  if (verbose) 
    {if (ok) cat("ok\n")
     else    cat("failed!\n")
    }

  if (verbose) cat("curvature test 2 ... ")

  global.assign("curve.test.bod", function( th) 
     {x <- matrix(c(1,2,3,4,5,7),6,1)
      y <- c( 8.3, 10.3, 19.0, 16.0, 15.6, 19.8) 
      y - ( th[1] * (1 - exp( - th[2] * x)))
     })
   on.exit(rm(curve.test.bod))
   r <- curve.Darray(genD("curve.test.bod",c(19.1430,  0.5311)),signif=signif, 
                show.extra.details=show.details)$stats
   r.BW   <- c(2,6, .05, 1.328,   0.184,      3.5,          0.49,        1.0, 1.01 )
   r.test <- c(2,6, .05, 1.32781, 0.18440417, 3.4990419358, 0.485941626, 1.0, 1.008372434 )
   err <- r -r.test
   max.error <- max(max.error, abs(err))
   ok <- all(abs(err) < fuzz.large )
   if (show.details | !ok) 
     {m <- rbind(r.BW, r.test, r, err)
      dimnames(m) <- list(
                    c("Bates&Watts","test value","calculated ", "difference "),
                    c(dimnames(r)[[2]]))
      print(m)
     cat("Above should compared with Bates and Watts table 7.2, p 257, model B, data set 11 (A1.4, p270.).\n")
     }
  all.ok <- all.ok & ok
  if (verbose) 
    {if (ok) cat("ok\n")
     else    cat("failed!\n")
    }



  if (verbose) cat("curvature test 3 ... ")

  global.assign("curve.test.isomerization", function(th)
  {x <- matrix(c(205.8,404.8,209.7,401.6,224.9,402.6,212.7,406.2,133.3,470.9,300.0,301.6
,297.3,314.0,305.7,300.1,305.4,305.2,300.1,106.6,417.2,251.0,250.3,145.1
,90.9,92.9,174.9,187.2,92.7,102.2,186.9,192.6,140.8,144.2,68.3,214.6
,142.2,146.7,142.0,143.7,141.1,141.5,83.0,209.6,83.9,294.4,148.0,291.0
,37.1,36.3,49.4,44.9,116.3,128.9,134.4,134.9,87.6,86.9,81.7,101.7
,10.5,157.1,86.0,90.2,87.4,87.0,66.4,33.0,32.9,41.5,14.7,50.2),   24,3)
   y <- c(3.541,2.397,6.6694,4.722,0.593,0.268,2.797,2.451,3.196,2.021,0.896,5.084,
5.686,1.193,2.648,3.303,3.054,3.302,1.271,11.648,2.002,9.604,7.754,11.590) 

 y - ((th[1]*th[3]*(x[,2]-x[,3]/1.632))/(1+x[,1]*th[2]+x[,2]*th[3]+x[,3]*th[4]))
})
   on.exit(rm(curve.test.isomerization,curve.test.bod))
   r <- curve.Darray(genD("curve.test.isomerization",
                           c(35.9200,0.0708,0.0377,0.1670)),signif=signif, 
                           show.extra.details=show.details)$stats
   r.BW   <-  c(4, 24, 0.05, 48.39,    0.048, 81.92, 0.081, 0.95,1.01)
   r.test <-  c(4, 24, 0.05, 46.00941, 0.048, 81.92, 0.081, 0.95,1.01)
   err <- r - r.test
   max.error <- max(max.error, abs(err))
   ok <- all(abs(err) < fuzz.large)
   if (ok) cat("Results check ok\n")
   else    cat("Results differ:\n")
   if (show.details | !ok) 
     {m <- rbind(r.BW, r.test, r, err)
      dimnames(m) <- list(
                    c("Bates&Watts","test value","calculated ", "difference "),
                    c(dimnames(r)[[2]]))
      print(m)
      cat("Above should compared with Bates and Watts table 7.2, p 257, model M, data set 32, Bates & Watts (data table A1.5, p271).\n")
      cat("\n Above test has never worked. The intrinsic curvature array differs and may need to be calculated analytically. The typical result is:\n")
      cat("calculated  4 24        0.05  4.600942e+01   0.045942003         77.891671     0.077777537           1.00     1.05959779 \n")
     }

  all.ok <- all.ok & ok
  if (verbose) 
    {if (ok) cat("ok\n")
     else    cat("failed!\n")
    }

  if (synopsis) 
    {if (verbose) cat("All curvature tests completed")
     if (all.ok) cat(" ok\n")
     else    
       {cat(", some failed!")
        if((!is.na(max.error)) && (max.error > fuzz.small))
            cat(" ( max. error magnitude= ", max.error,")")
        cat("\n")
       }
    }
  invisible(all.ok)
}


#######################################################################

#                    end

#######################################################################

#   Copyright 1993, 1994, 1995, 1996  Bank of Canada.
#   Copyright 1997 (June), Paul Gilbert.
#   Copyright 1997 (Aug.), Bank of Canada.
#   Copyright 1998, Bank of Canada.

#   The user of this software has the right to use, reproduce and distribute it.
#   Bank of Canada makes no warranties with respect to the software or its 
#   fitness for any particular purpose. The software is distributed by the Bank
#   of Canada and by Paul Gilbert solely on an "as is" basis. By using the  
#   software, user agrees to accept the entire risk of using this software.

################################################################################



#######################################################################

# These functions use functions in curve.hs

#######################################################################

#                            curvature calculation

#######################################################################

curve.TSestModel <- function (emodel, ...)  
 {D <-genD(emodel, ...)
  curve(D)
 }

#-----------------------------------------------------------------------

# S routines for calulating curvatures a la Bates and Watts.

# Notes:
#   Generating the D matrix can be computationally demanding. There are three
#   (or four) versions of the code for generating this matrix. The S version is slow 
#   and suffers from memory problems due to a bug in the way the current version of S
#   (S-PLUS Version 3.0 for SUN4) allocates memory in loops. The C version is
#   faster but suffers (even worse) from the memory problem. (A fix is promised 
#   in the next release of S-plus.) Both the S and the C versions take the name
#   of an S function as an arguement and call it. The fortran version is fast and
#   does not suffer from the memory problem, but works only with ARMA 
#   and KF models.
#    
#-----------------------------------------------------------------------

#######################################################################

#                               D matrix calculation

#######################################################################

genD.TSestModel <- function(estModel, ...)
   { invisible(genD( estModel$model, estModel$data, ...))}

#genD.TSmodel <- function(model, data, ...) {NextMethod("genD.TSmodel")}

genD.ARMA  <- function(model, data, d=0.01, eps=1e-4, r=6){
# Note: m,n,p have different meanings here than they do in 
#  time-series models! ms,ns,ps are use for time-series meaning
   n <-length(c(output.data(data))) # this has the same length as the residual
   sampleT <-periods(data) 
   ps <-dim(model$A)[3]
      ms <-dim(model$C)[3]
      ns <-ps  # this could be 1 except z0 is used for TREND and F, Q and R for scratch
     loc   <- match(model$location,       c("A","B","C","t"))
     cloc  <- match(model$const.location, c("A","B","C","t"))
   if(is.null(ms))
     {ms<-0
      C<-array(0,c(1,ns,1)) # can't call fortran with 0 length arrays
      u <- matrix(0,sampleT,1)
      G <- matrix(0,ns,1)
     }
   else
     {C <-model$C
      u <- input.data(data)
      G <- matrix(0,ns,ms)
     } 
   x <-model$parms
   zt <- 0.0001
   #Rbug zt <-unix.time(ARMA(model,data,sampleT,sampleT,result="like"))
   zt <- zt[1] *r*2*(length(x)*(length(x) + 3))/2
   if(zt>30) 
       cat("D matrix calculation very roughly estimated to take about ",
            (10*ceiling(zt/10)),"seconds(without other system activity)...\n")
   h0 <- abs(d*x)+eps*(x==0.0)
   D <- matrix(1e20, n,(length(x)*(length(x) + 3))/2) 
   Daprox <-  matrix(0,n,r) 
   Hdiag  <-  matrix(0,n,length(x))
   Haprox <-  matrix(0,n,r)
   storage.mode(D)     <-"double"
   storage.mode(Daprox)<-"double"
   storage.mode(Haprox)<-"double"
   storage.mode(Hdiag )<-"double"
   D <-.Fortran("gend",
            D=D,
            as.integer(is.ARMA(model)), 
            p=as.integer(length(x)),
            x0=as.double(x),
            as.double(h0),
            as.integer(n),    #6
            as.integer((length(x)*(length(x)+ 3))/2), #cols of D
            f0=double(n),      
            as.integer(r),
            #                       work space for GEND
            Haprox=Haprox,     #10   
            Hdiag=Hdiag,         
            Daprox=Daprox,        
            x=double(length(x)),
            delta=double(length(x)),
            f1=double(n),
            f2=double(n),
            #                   work space for ARMAp / KFp
         #     cov=matrix(1e20,ps,ps),       
            # pred is f0,f1,f2 passed above
            as.integer(ms),     #  input dimension m  #17
            as.integer(ps),     # output dimension p 
            as.integer(sampleT),   
            as.integer(periods(data)), 
            u=as.double(u), 
            y=as.double(output.data(data)),   
            #   model$parm is passed above as x (it is the parameter for curvature calculation)   
            as.integer(loc),   #as.character(model$location), #23
            as.integer(model$i),
            as.integer(model$j),
            as.integer(length(model$const)),
            const=as.double(model$const),
            as.integer(cloc),  #as.character(model$const.location),
            as.integer(model$const.i ), 
            as.integer(model$const.j),
        #  for ARMA models:
            as.integer(model$l),       #31
            as.integer(model$const.l),
            as.integer( dim(model$A)[1]),#1+order of A
            as.integer( dim(model$B)[1]),#1+order of B
            as.integer( dim(C)[1]),#1+order of C
            A=model$A,   
            B=model$B,  
            C=C, 
        #  for state space models:
            as.integer(ns),  # state dim.  #39
        #    state=as.double(matrix(0,sampleT,ns)),  
        #    track=as.double(array(0,c(sampleT,ns,ns))),  
            z0=as.double(rep(0,ps)), # note: this is TREND for ARMA
            P0=as.double(diag(0,ps)),
            F=as.double(matrix(0,ns,ns)),  
            G=as.double(G),  
            H=as.double(matrix(0,ps,ns)),  
            K=as.double(matrix(0,ns,ps)),  
            Q=as.double(matrix(0,ns,ns)),  
            R=as.double(matrix(0,ps,ps)), 
            gain=as.logical(F)   #48
            )[c("D","p","f0")]
   class(D) <- "Darray"
   invisible(D)
}

#Rbug this does not seem to work as genD.KF or genD.KF.innov

genD.innov <- function(model, data, d=0.01, eps=1e-4, r=6){
# Note: m,n,p have different meanings here than they do in 
#  time-series models! ms,ns,ps are use for time-series meaning
   n <-length(c(output.data(data))) # this has the same length as the residual
   sampleT <-periods(data) 
   ns <-dim(model$F)[2]
   ps <-dim(model$H)[1]
   ms <-dim(model$G)[2]
   if(is.null(ms))
     {ms<-0
      C<-array(0,c(1,ns,1)) # can't call fortran with 0 length arrays
      u <- matrix(0,sampleT,1)
      G <- matrix(0,ns,1)
     }
   else
     {C <- array(0,c(1,ns,ms))
      u <- input.data(data)
      G <- matrix(0,ns,ms)
     } 
   x <-model$parms
   zt <- 0.0001
   #Rbug zt <-unix.time(KF(model,data,sampleT,sampleT,result="like"))
   zt <- zt[1] *r*2*(length(x)*(length(x) + 3))/2
   if(zt>30) 
       cat("D matrix calculation very roughly estimated to take about ",
           (10*ceiling(zt/10)),"seconds(without other system activity)...\n")
   h0 <- abs(d*x)+eps*(x==0.0)
   D <- matrix(1e20, n,(length(x)*(length(x) + 3))/2) 
   Daprox <-  matrix(0,n,r) 
   Hdiag  <-  matrix(0,n,length(x))
   Haprox <-  matrix(0,n,r)
   storage.mode(D)     <-"double"
   storage.mode(Daprox)<-"double"
   storage.mode(Haprox)<-"double"
   storage.mode(Hdiag )<-"double"
   loc   <- match(model$location,       c("f","G","H","K","Q","R","z","P"))
   cloc  <- match(model$const.location, c("f","G","H","K","Q","R","z","P"))
   D <-.Fortran("gend",
            D=D,
            as.integer(is.ARMA(model)), 
            p=as.integer(length(x)),
            x0=as.double(x),
            as.double(h0),
            as.integer(n),    #6
            as.integer((length(x)*(length(x)+ 3))/2), #cols of D
            f0=double(n),      
            as.integer(r),
            #                       work space for GEND
            Haprox=Haprox,        
            Hdiag=Hdiag,         
            Daprox=Daprox,        
            x=double(length(x)),
            delta=double(length(x)),
            f1=double(n),                # 15
            f2=double(n),
            #                   work space for ARMAp / KFp       
            # cov=matrix(1e20,ps,ps),       
            # pred is f0,f1,f2 passed above
            as.integer(ms),     #  input dimension m  
            as.integer(ps),     # output dimension p 
            as.integer(sampleT),   
            as.integer(periods(data)), 
            u=as.double(u), 
            y=as.double(output.data(data)),   #22
            #   model$parm is passed above as x (it is the parameter for curvature calculation)   
            as.integer(loc),   #as.character(model$location) bug
            as.integer(model$i),
            as.integer(model$j),
            as.integer(length(model$const)),
            const=as.double(model$const),     #28
            as.integer(cloc),  #as.character(model$const.location),
            as.integer(model$const.i ), 
            as.integer(model$const.j),
        #  for ARMA models:
            as.integer(loc),                   #31
            as.integer(loc),
            as.integer(1),#1+order of A
            as.integer(1),#1+order of B
            as.integer(1),#1+order of C
            A=as.double(array(0,c(1,ps,ps))),   
            B=as.double(array(0,c(1,ps,ps))),  
            C=as.double(C), 
        #  for state space models:
            as.integer(ns),  # state dim.     #39
        #    state=as.double(matrix(0,sampleT,ns)),  
        #    track=as.double(array(0,c(sampleT,ns,ns))),  
            z0=as.double(rep(0,ns)),  
            P0=as.double(diag(0,ps)),
            F=as.double(matrix(0,ns,ns)),  
            G=as.double(G),  
            H=as.double(matrix(0,ps,ns)),  
            K=as.double(matrix(0,ns,ps)),  
            Q=as.double(matrix(0,ns,ns)),  
            R=as.double(matrix(0,ps,ps)),
            gain=as.logical(is.innov.SS(model)) #48
            )[c("D","p","f0")]
   class(D) <- "Darray"
   invisible(D)
}

#######################################################################

#              span (dimension of tangent space) calculation

#######################################################################

span.TSestModel <- function (emodel, fortran=T, ...)  
 {# calculate the singular values of the tangents
  # the fortran version calculates the whole D matrix (which seems like
  # a waste, but the fortran is much faster, so ...
  if (fortran)
   {D <- genD(emodel, ... )$D[,seq(length(parms(emodel))),drop=F]
    return(svd(D)$d)
   }
  else
   span.default("residual", emodel$model$parms,
               func.args=list(model=emodel$model, data=emodel$data), ...)
 }


#######################################################################

#            calculate Fisher Info (Hessian of the likelihood)

#######################################################################

hessian.TSestModel <- function (emodel, ...)  
 {# like returns neg. log likelihood
  hessian.default("like", emodel$model$parms,
       func.args=append(list(model=emodel$model, data=emodel$data), list(...)))
 }


#######################################################################

# Test routines for calulating curvatures.

#######################################################################
 

dsecurve.function.tests <- function( verbose=T, synopsis=T, fuzz.small=1e-12, fuzz.large=1e-6, show.details=F, show.extra.details=F)
{# Tests of DSE curvature functions

# comparison values come only from first run of code.

  random.number.test()
  test.seed <- set.seed(c( 979, 1479, 1542))

  max.error <- 0
  all.ok <- T
  signif <- 0.05
  if (synopsis & !verbose) cat("All curvature tests ...")

  if (verbose) cat("DSE curvature test 1 ... ")

# simplified from user guide
  VARmodel<-list(A=array(c(1,.5,.3),c(3,1,1)),
                 B=array(1,c(1,1,1)),
                 C=NULL, description="simplified guide example")
  class(VARmodel)<-c("ARMA","TSmodel")
  VARmodel <-set.parameters(VARmodel)
  VARmodel <-l(VARmodel,simulate(VARmodel, seed=test.seed))

# unstable model
  VARmodel2<-list(A=array(c(1,-0.5,-0.5),c(3,1,1)),
                 B=array(1,c(1,1,1)),
                 C=NULL, description="simplified guide example")
  class(VARmodel2)<-c("ARMA","TSmodel")
  VARmodel2 <-set.parameters(VARmodel2)
  VARmodel2 <-l(VARmodel2,simulate(VARmodel2, seed=test.seed))
# Mod(roots(VARmodel, by.poly=T))
# [1] 0.5477226 0.5477226
# Mod(roots(VARmodel2, by.poly=T))
# [1] 1.0 0.5

# Rbug change in rnorm from 0.16.1 to 0.49
#VARmodel$estimates$like[1]
#   0.49     1.4319082716463254e+02
#   0.16.1   1.400873455714374e+02
# VARmodel$data$noise$rv100[1:5]
#   0.49   [1] 0.56933541 0.10513578 0.05846933 0.07537961 0.70437349
#   0.16.1 [1] 0.56933541 0.10513578 0.05846933 0.07537961 0.70437349
# VARmodel$data$noise$w0
#   0.16.1 [1,] -0.1349460
#   0.49   [1,] -0.555939
# VARmodel$data$noise$w[1:5]
#   0.16.1 [1]  0.2262262  0.1075778  0.1503252 -0.4913841 -1.1676293
#   0.49   [1] -1.9024311  1.5245959 -0.7863495  0.1328128 -1.8405500


  if (fuzz.small < abs(VARmodel$estimates$like[1]- 1.431908271646325e+02))
     warning("model  likelihood  does not correspond to expected test value.")
 
  if( fuzz.small < abs(sum(VARmodel$data$noise$w)+ 2.928188921879759)) #0.49beta
      warning("Check sum of noise does not correspond to expected test value.")
      #VARmodel$data$noise$w0  -5.559389931781886e-01
  SSmodel  <- l(to.SS(VARmodel),  VARmodel$data)
  ARMAmodel<- l(to.ARMA(SSmodel), VARmodel$data)
  if (fuzz.small < abs(ARMAmodel$estimates$like[1]- 1.431908271646325e+02))
     warning("model likelihood does not correspond to expected test value.")

  spanVAR <- span(VARmodel, fortran=F)
  ok <- fuzz.small > max(abs( spanVAR - 
                        c( 14.48076491702112, 8.491923097006795))) # 0.49


  spanVAR.f <- span(VARmodel, fortran=T)
  ok <- ok & fuzz.small >  max(abs( spanVAR- spanVAR.f))
  all.ok <- all.ok & ok
  if (verbose) 
    {if (ok) cat("ok\n")
     else    cat("failed!\n")
    }


  if (verbose) cat("DSE curvature test 2 ... ")
 
  spanSS.f <- span(SSmodel, fortran=T)
  ok <- (fuzz.small > max(abs( spanSS.f - c( 
  11.60515753778120, 9.510768052278456, 5.128454728058138, 1.868858671602684)))) 
  spanSS <- span(SSmodel, fortran=F)
  ok <- ok & fuzz.small >  max(abs( spanSS- spanSS.f))
 
  all.ok <- all.ok & ok
  if (verbose) 
    {if (ok) cat("ok\n")
     else    cat("failed!\n")
    }


  if (verbose) cat("DSE curvature test 3 ... ")

  spanARMA.f <- span(ARMAmodel, fortran=T)
  ok <- fuzz.small > max(abs( spanARMA.f -
                      c( 14.48076491702112, 8.491923097006794)))

  spanARMA <- span(ARMAmodel, fortran=F)
  ok <- ok & fuzz.small >  max(abs( spanARMA- spanARMA.f))
 
  all.ok <- all.ok & ok
  if (verbose) 
    {if (ok) cat("ok\n")
     else    cat("failed!\n")
    }


  if (verbose) cat("DSE curvature test 4 ...skipping part ")
# calculating from the score using genD might be quicker.
  hessianVAR <- hessian(VARmodel) 
# cat("hessianVAR  eigenvalues ",eigen(hessianVAR)$values, "\n")
  ok <- fuzz.large > max(abs( hessianVAR -
       c( -20.91669740007358, 9.256369489592716,
           9.256369489592716, 130.3640431536360))) #0.49
#       c( -2.8650443811673960, -5.7586504058931549,
#          -5.7586504058931549, 1.3435842667006622e+02)))

   hessianSS <- hessian(SSmodel)
# cat("hessianSS  eigenvalues ",eigen(hessianSS)$values, "\n")
#   ok <- ok & (fuzz.small > max(abs(hessianSS - c( 1.9175724722710770, 
# -5.2761201996604212,  8.1411645809153104,
#  3.2449965474253240e+01, -5.2761201996604212,  3.8410779336861087, 
# 9.2870888235186815,  1.9432914053685220,  8.1411645809153104, 
# 9.2870888235186815,  1.4043059601391144e+01, -5.6520720360307855e+01, 
# 3.2449965474253240e+01,  1.9432914053685220, -5.6520720360307855e+01, 
# 1.1965135248989206e+02) )) )

  ok <- ok & (fuzz.small > max(abs(hessianSS - 
       c( -3.596028414140290,  2.152671343177936,  1.876402605694525e+01,
      1.900444044688388e+01,  2.152671343177936, -5.660341075640381,
      2.296101908166416e+01,  6.783471711016179,  1.876402605694525e+01,
      2.296101908166416e+01, -4.300418440719920, -3.505125931129112e+01,
      1.900444044688388e+01,  6.783471711016179, -3.505125931129112e+01,
      9.723363149618343e+01))))  # R 0.49
#       c( 1.917572472307711e+00,
#       -5.276120199771213e+00, 8.141164580892589e+00, 3.244996420384564e+01,
#       -5.276120199771213e+00, 3.841077933868319e+00, 9.287089214414681e+00,
#        1.943293362001617e+00, 8.141164580892589e+00, 9.287089214414681e+00,
#        1.404305913622460e+01,-5.652071918242696e+01, 3.244996420384564e+01,
#        1.943293362001617e+00,-5.652071918242696e+01, 1.196513481792662e+02))))


#   hessianARMA <- hessian(ARMAmodel)
#  ok <- ok & (fuzz.small > max(abs( hessianARMA - )) )

  all.ok <- all.ok & ok
  if (verbose) 
    {if (ok) cat("ok\n")
     else    cat("failed!\n")
    }


  if (verbose) cat("DSE curvature test 5 ... ")

  curveVAR <- curve(VARmodel)$stats
  ok <-fuzz.small> max(abs(curveVAR - 
   c(2, 100, 0.05, 1.673308268356802e-10, 9.014674004133696e-10, 
     2.941028153763199e-10, 1.584430707988419e-09, 1, 1)))  # R 0.49
#   c(2,100,0.05, 1.399390438768164e-10,  7.580201220157363e-10, 
#     2.459586650202710e-10, 1.332305924811175e-09, 1, 1)))

# and for comparison
  curveVAR.def <- curve.default("residual", parms(VARmodel), 
                     func.args=list(model=VARmodel$model, data=VARmodel$data),
                     d=0.01, eps=1e-4,r=6, show.details=F)$stats
  ok <- ok & fuzz.large > max(abs(curveVAR - curveVAR.def))

  all.ok <- all.ok & ok
  if (verbose) 
    {if (ok) cat("ok\n")
     else    cat("failed!\n")
    }


  if (verbose) cat("DSE curvature test 6 ... ")

  curveSS <- curve(SSmodel)$stats
  ok <-fuzz.small > max(abs(curveSS - 
     c(4, 100, 0.05, 4.953258611615619e-01, 4.660722059607809e-01,
    7.779102253761303e-01, 7.31967303161349e-01,  1, 1.000000000000251))) #0.49
#    c(4,100,0.05, 5.304467080568718e-01, 4.840339818265660e-01, 
#    8.330675835234724e-01, 7.601763069859021e-01, 1, 1.0000000000001967)))

  curveSS.def <- curve.default("residual", parms(SSmodel), 
                     func.args=list(model=SSmodel$model, data=SSmodel$data),
                     d=0.01, eps=1e-4,r=6, show.details=F)$stats

if (any(is.na(curveSS.def))) 
       cat("Invalid value. skipping comparison. Not ")
else   ok <- ok & fuzz.small > max(abs(curveSS - curveSS.def))

  all.ok <- all.ok & ok
  if (verbose) 
    {if (ok) cat("ok\n")
     else    cat("failed!\n")
    }


  if (verbose) cat("DSE curvature test 7 ... ")


  curveARMA <- curve(ARMAmodel)$stats
  ok <- fuzz.small > max(abs(curveARMA - curveVAR))
  curveARMA.def <- curve.default("residual", parms(ARMAmodel), 
                     func.args=list(model=ARMAmodel$model, data=ARMAmodel$data),
                     d=0.01, eps=1e-4,r=6, show.details=F)$stats

  ok <- ok & fuzz.large > max(abs(curveARMA - curveARMA.def))

  all.ok <- all.ok & ok
  if (verbose) 
    {if (ok) cat("ok\n")
     else    cat("failed!\n")
    }


# from user guide

  if (verbose) cat("DSE curvature test 8 ... ")

  test.seed <- c( 979, 1479, 1542)
  VARmodel<-list(A=array(c(1,.5,.3,0,.2,.1,0,.2,.05,1,.5,.3),c(3,2,2)),
             B=array(c(1,.2,0,.1,0,0,1,.3),c(2,2,2)),
            ,C=NULL)
  class(VARmodel)<-c("ARMA","TSmodel")
  VARmodel<-set.parameters(VARmodel)
  VARmodel<-l(VARmodel,simulate(VARmodel, seed=test.seed))
  SSmodel  <- l(to.SS(VARmodel),  VARmodel$data)
  ARMAmodel<- l(to.ARMA(SSmodel), VARmodel$data)

  spanVAR.f <- span(VARmodel, fortran=T) 
  ok <- fuzz.small > max(abs( spanVAR.f - 
     c( 1.783668762976735e+01, 1.676713740212845e+01, 1.622828962916468e+01,
     1.336260553115620e+01, 1.032677905977513e+01, 9.930700976937008,
     8.881377672535175, 8.213398828201516, 2.220296617939313,
     1.914140078418723, 1.483876742824580)))  #R 0.49
#     c( 18.48351551541406, 16.77792221175794, 14.76451878225780, 
#        11.73419485641150, 1.091664124276006e+01 ,1.062215333373193e+01, 
#        8.779077882383611, 8.038279565359707, 2.264211498425161,
#        1.995568875481004, 1.499435329458541)))
# print(spanVAR.f, digits=16)
 
  spanVAR <- span(VARmodel, fortran=F) 
  ok <- ok & fuzz.small >  max(abs( spanVAR- spanVAR.f))
 
  all.ok <- all.ok & ok
  if (verbose) 
    {if (ok) cat("ok\n")
     else    cat("failed!\n")
    }


  if (verbose) cat("DSE curvature test 9 ... ")

  spanSS <- span(SSmodel, fortran=T)
  ok <- fuzz.small > max(abs( spanSS -
      c( 1.431917078200739e+01, 1.275539322641331e+01, 1.203995590448420e+01,
     1.117807502484729e+01, 8.764237474316120, 8.411065047433565,
     8.111923814106442,     7.711205445427577, 4.581684918879841,
     4.492812448867578,     3.090950568240565, 2.779087870909759,
     2.290116662046349,     2.121173644211920, 1.823111321101024,
     1.736460343300366)))  # 0.49
#  print(spanSS, digits=16)

  spanSS <- span(SSmodel, fortran=F) 
  ok <- ok & fuzz.small >  max(abs( spanVAR- spanVAR.f))
 
  all.ok <- all.ok & ok
  if (verbose) 
    {if (ok) cat("ok\n")
     else    cat("failed!\n")
    }


  if (verbose) cat("DSE curvature test 10... ")

  spanARMA.f <- span(ARMAmodel, fortran=T)
  ok <- fuzz.small > max(abs( spanARMA.f - 
c( 3.011169300794247e+01, 2.690373853859740e+01, 2.194916692782002e+01,
   1.941211883487011e+01, 1.702384984930141e+01, 1.505916731276925e+01,
   1.219808033127580e+01, 1.120321261421515e+01, 1.053442404016652e+01,
   9.761163478425873e+00, 6.772415827211282e+00, 6.077275381439208e+00,
   5.537891818483530e+00, 5.225555659106909e+00, 2.702083018243574e+00,
   2.543373410280471e+00, 7.477834552669831e-01, 4.619958604596496e-01,
   2.038270252146126e-06, 2.556984253017427e-07, 1.795257106881694e-15,
   7.973651962079763e-17))) #R0.49
# c( 2.768304162846298e+01, 2.393941491091734e+01, 1.988022205000436e+01, 
#1.822550017859639e+01, 1.630502757392022e+01, 1.461512246650834e+01, 
#1.275451631155268e+01, 1.221047501548486e+01, 1.073726354564602e+01, 
#1.026204415323931e+01, 7.265742818227836e+00, 6.282013694588179e+00, 
#5.974725282653903e+00, 5.251293695221127e+00, 2.860581954100438e+00, 
#2.471994703685991e+00, 7.121798982572878e-01, 4.406741938800671e-01, 
#1.433057563479431e-15, 9.026520997009618e-16, 4.800022955996391e-17, 
#0.000000000000000e+00 )))
#  print(spanARMA.f, digits=16)

  spanARMA <- span(ARMAmodel, fortran=F) 
  ok <- ok & fuzz.small >  max(abs( spanARMA- spanARMA.f))
 
  all.ok <- all.ok & ok
  if (verbose) 
    {if (ok) cat("ok\n")
     else    cat("failed!\n")
    }

ARMAmodel.fixed <- l(fix.constants(ARMAmodel), VARmodel$data)

spanARMA.fix <- span(ARMAmodel.fixed)

  if (verbose) cat("DSE curvature test 11... ")

  hess <- hessian(VARmodel)  
  ok <- fuzz.small > abs(sum(hess) - 1.564863209508781e+03)
  if (verbose) cat("DSE curvature test 11b... ")
  hess <- hessian(SSmodel)
  ok <- ok & (fuzz.small > abs(sum(hess) - 1.447334905214157e+03))
  if (verbose) cat("DSE curvature test 11c... ")
  hess <- hessian(ARMAmodel)
  ok <- ok & (fuzz.large > abs(sum(hess) - 1.4707657251903214e+04))
  if (verbose) cat("DSE curvature test 11d... ")
  hess <- hessian(ARMAmodel.fixed)
  ok <- ok & (fuzz.large > abs(sum(hess) - 3.425881557792753e+03))
  all.ok <- all.ok & ok
  if (verbose) 
    {if (ok) cat("ok\n")
     else    cat("failed!\n")
    }


  if (verbose) cat("DSE curvature test 12... ")

  curveVAR <- curve(VARmodel)$stats
if (any(is.na(curveVAR))) 
       cat("Invalid value. skipping comparison. Not ")
else     ok <- fuzz.small > max(abs(curveVAR - 
       c( 11, 200, 0.05, 3.238933406886928e-01, 2.653669818330267e-01,
        4.393018981040493e-01, 3.599216290323103e-01, 1,1.000000000000483e+00)))
#       c( 11, 200, 0.05,  .304793372284, .253725786529, 
#             0.41339629487, .344132483190, 1, 1 )))

  curveVAR.def <- curve.default("residual", parms(VARmodel), 
                     func.args=list(model=VARmodel$model, data=VARmodel$data),
                     d=0.01, eps=1e-4,r=6, show.details=F)$stats

if (any(is.na(curveVAR.def))) 
       cat("Invalid value. skipping comparison. Not ")
else     ok <- ok & fuzz.small > max(abs(curveVAR - curveVAR.def))

  all.ok <- all.ok & ok
  if (verbose) 
    {if (ok) cat("ok\n")
     else    cat("failed!\n")
    }


  if (verbose) cat("DSE curvature test 13... ")

  curveSS <- curve(SSmodel)$stats  

  curveARMA.fixed <- curve(ARMAmodel.fixed)$stats

  curveARMA <- curve(ARMAmodel)$stats #singular matrix in solve(R11)
# D generated. Calculating curvature.Warning: eliminating degenerate subspace for R11.
# Warning: B is not symmetric.
# Warning: eigen for non-symmetric matrix is fictitious
# N.B. (I-B) is not positive definite as it should be at a local min! 
#    ARMAX will calculate to NA 
# Eigenvalues of B [1] -1.098230e+47 -5.735942e+15 -1.898362e+13 -5.863185e+01 -1.398720e+00
#  [6] -8.339023e-01 -2.658607e-01 -7.732492e-02 -6.136313e-02 -2.463111e-02
# [11] -2.361107e-03  9.469655e-03  9.208593e-02  2.977150e-01  6.957031e-01
# [16]  1.544891e+00  7.220152e+00  2.266144e+01  1.865402e+13  9.217375e+30
# [21]  5.552606e+45  1.105510e+47
# Warning: NAs produced in function "sqrt"
# >curveARMA
#       P   N Sign. level RMS Parameter RMS Intrinsic c*sqrt(F) Parameter
# [1,] 22 200        0.05  1.624603e+48  4.083931e+48        2.056528e+48
#      c*sqrt(F) Intrinsic Min Axis Ratio Max Axis Ratio
# [1,]        5.169704e+48       1.001183             NA
 
  ok <- fuzz.small > max(abs(curveARMA - curveVAR))
 print(curveARMA, digits=18)
 print(curveVar,  digits=18)
  all.ok <- all.ok & ok
  if (verbose) 
    {if (ok) cat("ok\n")
     else    cat("failed!\n")
    }




# following from dse1 tests are much slower

  if (verbose) cat("DSE curvature test 14 ... ")
  VARmodel <- est.VARX.ls(example.BOC.93.4.data.all, re.add.means=F)
  SSmodel  <- l(to.SS(VARmodel),  example.BOC.93.4.data.all)
  ARMAmodel<- l(to.ARMA(SSmodel), example.BOC.93.4.data.all)

  spanVAR <- span(VARmodel)

  if (verbose) cat("DSE curvature test 15 ... ")
  spanSS <- span(SSmodel)
  if (verbose) cat("DSE curvature test 16 ... ")
  spanARMA <- span(ARMAmodel)

  if (verbose) cat("DSE curvature test 17 ... ")
  curveVAR <- curve(VARmodel)$stats
  if (verbose) cat("DSE curvature test 18 ... ")
  curveSS <- curve(SSmodel)$stats
  if (verbose) cat("DSE curvature test 19 ... ")
  curveARMA <- curve(ARMAmodel)$stats

  if (verbose) cat("DSE curvature test 20 ... ")
  hessianVAR <- hessian(VARmodel)
  if (verbose) cat("DSE curvature test 21 ... ")
  hessianSS <- hessian(SSmodel)
  if (verbose) cat("DSE curvature test 22 ... ")
  hessianARMA <- hessian(ARMAmodel)

  all.ok <- all.ok & ok
  if (verbose) 
    {if (ok) cat("ok\n")
     else    cat("failed!\n")
    }


  if (synopsis) 
    {if (verbose) cat("All curvature tests completed")
     if (all.ok) cat(" ok\n")
     else    
       {cat(", some failed!")
        if((!is.na(max.error)) && (max.error > fuzz.small))
            cat(" ( max. error magnitude= ", max.error,")")
        cat("\n")
       }
    }
  invisible(all.ok)
}


#######################################################################

#                    end

#######################################################################

#   Copyright 1993, 1994, 1995, 1996  Bank of Canada.
#   Copyright 1997 (June), Paul Gilbert.
#   Copyright 1997 (Aug.), Bank of Canada.
#   Copyright 1998, Bank of Canada.

#   The user of this software has the right to use, reproduce and distribute it.
#   Bank of Canada makes no warranties with respect to the software or its 
#   fitness for any particular purpose. The software is distributed by the Bank
#   of Canada and by Paul Gilbert solely on an "as is" basis. By using the  
#   software, user agrees to accept the entire risk of using this software.

################################################################################



# retrieve data from file eg1.dat
# Define some example generic model evaluation functions and
# define data verification functions and
# a suite of test functions (for DSE functions)
# These can be executed by
#    example.verify.data()
#    example.verify.raw.data()
#    example.tests()

example.get.eg.raw.data <-function(file)
{ example.raw.data <- t(matrix(scan(file), 5, 364))[, 2:5]
  example.raw.data <-list(
    input= ts(example.raw.data[,1,drop = F],
       start=c(1961,3), frequency=12),
    output=ts(example.raw.data[, 2:4, drop = F],
       start=c(1961,3), frequency=12) )
  dimnames(example.raw.data$input) <- list(NULL, "R90")
  dimnames(example.raw.data$output) <- list(NULL, c("M1","GDPl2", "CPI"))
  class(example.raw.data) <-"TSdata"

  example.raw.data
}

example.convert.eg.data <-function(example.raw.data)
{ example.data <-list(
    input =ts(input.data(example.raw.data)[2:364, , drop = F], 
       start=c(1961,3), frequency=12),
    output=ts(diff(log(output.data(example.raw.data)[,, drop = F])),
       start=c(1961,3), frequency=12) )
  dimnames(example.data$input) <- list(NULL, "R90")
  dimnames(example.data$output) <- list(NULL, c("M1","GDPl2", "CPI"))
  class(example.data) <-"TSdata"
  example.data
}

example.verify.data <- function(example.data, verbose=T, fuzz.small=1e-14)
{# verify that data in example.data is correct data.
  z <-cbind(input.data(example.data),output.data(example.data))
  if (verbose) cat("example.data\n sample size...")
  ok <- 363 == dim(z)[1]
  ok <- ok & ( 4 == dim(z)[2] )
  if (verbose) 
    {if (ok) cat("ok\n")
     else cat("NOT CORRECT!\n")
    }
  if (verbose) cat(" sample mean...")
  ok <- ok & all(fuzz.small > abs(apply(z, 2, mean) -
           c(8.606143250688707, 0.005502904131200907, 0.003297288176061855,
              0.004576853034062842)))
  if (verbose) 
    {if (ok) cat("ok\n")
     else cat("NOT CORRECT!\n")
    }

  if (verbose) cat(" sample var ...")
  ok <- ok & all(fuzz.small > abs(apply(z, 2, var) -
             c(12.5442812169916, 0.0001384487711077435, 3.572716474599408e-05,
                 1.396066119144909e-05)))
  if (verbose) 
    {if (ok) cat("ok\n")
     else cat("NOT CORRECT!\n")
    }
invisible(ok)
}

example.verify.raw.data <- function(example.raw.data, verbose=T, fuzz.small=1e-5)
{# verify that data in example.raw.data is correct data.
  z <-cbind(input.data(example.raw.data),output.data(example.raw.data))
  if (verbose) cat("example.raw.data\n sample size...")
  ok <- 364 == dim(z)[1]
  ok <- ok & ( 4 == dim(z)[2] )
  if (verbose) 
    {if (ok) cat("ok\n")
     else cat("NOT CORRECT!\n")
    }
  if (verbose) cat(" sample mean...")
  ok <- ok & all(fuzz.small > abs(apply(z, 2, mean) -c(8.592884615384618, 19217.14560439561, 329471.3387362638, 58.71483516483516)))
  if (verbose) 
    {if (ok) cat("ok\n")
     else cat("NOT CORRECT!\n")
    }

  if (verbose) cat(" sample var ...")
  ok <- ok & all(fuzz.small > abs(apply(z, 2, var) -c(12.57371204174613, 125360567.0779145, 11249376720.93049, 1067.923691157328)))
  if (verbose) 
    {if (ok) cat("ok\n")
     else cat("NOT CORRECT!\n")
    }
invisible(ok)
}

example.like.sub.sample <- function(model)
{#  example of generic evaluation using class and methods approach.
 # this example function takes a TSmodel and evaluates it using a 
 # fixed data set (the first 240 observations from example.data).
 l(model, example.data, sampleT=240, predictT=240)
}

example.like.total.sample <- function(model)
{#  example of generic evaluation using class and methods approach.
 # this example function takes a TSmodel and evaluates it using a 
 # fixed data set (example.data).
 l(model, example.data, sampleT=363, predictT=363)
}

example.tests <- function(example.data, verbose=T, summary=T,
        fuzz.small=1e-14, fuzz.large=1e-8)
{# A short set of tests of DSE functions using example.data.
 # Use as follows:
 #   example.tests(example.BOC.93.4.data.all) 
 # The main short coming of these tests is that they do not test
 # functions which produce output, such as display, summary, graph
 # and check.residuals.
 # Note- The first part of these tests uses the total sample. 
 #    Working paper estimated with sub-sample!

  if (!example.verify.data(example.data,fuzz.small=fuzz.small, verbose=verbose))
     stop("example.data does not verify correctly. Example testing stopped.")
  max.error <- NA
  if (verbose) cat("Testing DSE functions with example.data.\n")
  if (verbose) cat(" est.VARX.ar...")
  VARmodel <- est.VARX.ar(example.data, re.add.means=F)
  error <- abs(-3879.7321062329338019 - VARmodel$estimates$like[1])
  ok <- fuzz.large > error
  if (!ok) {if (is.na(max.error)) max.error <- error
            else max.error <- max(error, max.error)}
  all.ok <- ok 
  if (verbose) 
    {if (ok) cat("ok\n")
     else cat("NOT CORRECT!\n")
    }

  if (verbose) cat(" est.VARX.ls...")
  error <- abs(-4125.055726045400661 - 
                est.VARX.ls(example.data)$estimates$like[1])
  ok <- fuzz.large > error
  if (!ok) {if (is.na(max.error)) max.error <- error
            else max.error <- max(error, max.error)}
  all.ok <- all.ok & ok 
  if (verbose) 
    {if (ok) cat("ok\n")
     else cat("NOT CORRECT!\n")
    }

  if (verbose) cat("       to.SS...")
  SSmodel <- l(to.SS(VARmodel),example.data)
  error <- abs(VARmodel$estimates$like[1]-SSmodel$estimates$like[1])
  ok <- fuzz.large > error
  if (!ok) {if (is.na(max.error)) max.error <- error
            else max.error <- max(error, max.error)}
  all.ok <- all.ok & ok 
  if (verbose) 
    {if (ok) cat("ok\n")
     else cat("NOT CORRECT!\n")
    }

  if (verbose) cat("     to.ARMA...")
  ARMAmodel <- l(to.ARMA(SSmodel),example.data)
  error <- abs(VARmodel$estimates$like[1]-ARMAmodel$estimates$like[1])
  ok <- fuzz.large > error
  if (!ok) {if (is.na(max.error)) max.error <- error
            else max.error <- max(error, max.error)}
  all.ok <- all.ok & ok 
  if (verbose) 
    {if (ok) cat("ok\n")
     else cat("NOT CORRECT!\n")
    }

  if (verbose) cat(" McMillan.degree...")
  ok <- 12 == McMillan.degree(VARmodel$model, verbose=F)$distinct
  all.ok <- all.ok & ok 
  if (verbose) 
    {if (ok) cat("ok\n")
     else cat("NOT CORRECT!\n")
    }

  if (verbose) cat(" Working Paper 93-4 comparisons:\n")
  if (verbose) cat("      VAR model likelihood...")
  sub.sample <- window(example.data,end=c(1981,2))
  VARmodel <- est.VARX.ar(sub.sample, re.add.means=F)
  error <- abs(-2567.3280114943772787 - VARmodel$estimates$like[1])
  ok <- fuzz.large > error
  if (!ok) {if (is.na(max.error)) max.error <- error
            else max.error <- max(error, max.error)}
  all.ok <- all.ok & ok 
  if (verbose) 
    {if (ok) cat("ok\n")
     else cat("NOT CORRECT!\n")
    }

  if (verbose) cat("      VAR model roots     ...")
  error <- abs(4.6786105186422091151 - sum(Mod(roots(VARmodel, by.poly=T))))
  ok <- fuzz.small > error
  if (!ok) {if (is.na(max.error)) max.error <- error
            else max.error <- max(error, max.error)}
  all.ok <- all.ok & ok 
  if (verbose) 
    {if (ok) cat("ok\n")
     else cat("NOT CORRECT!\n")
    }


  if (verbose) cat("      SS  model likelihood...")
  SS1.model <- l(balance.Mittnik(to.SS(VARmodel), n=9),sub.sample)
  error <- abs(-2567.328011494376824 - SS1.model$estimates$like[1])
  ok <- fuzz.large > error
  if (!ok) {if (is.na(max.error)) max.error <- error
            else max.error <- max(error, max.error)}
  all.ok <- all.ok & ok 
  if (verbose) 
    {if (ok) cat("ok\n")
     else cat("NOT CORRECT!\n")
    }

  if (verbose) cat("      SS  model roots     ...")
  error <- abs(4.6786105186422082269 - sum(Mod(roots(SS1.model))))
  ok <- fuzz.small > error
  if (!ok) {if (is.na(max.error)) max.error <- error
            else max.error <- max(error, max.error)}
  all.ok <- all.ok & ok 
  if (verbose) 
    {if (ok) cat("ok\n")
     else cat("NOT CORRECT!\n")
    }

  if (verbose) cat("     ARMA model likelihood...")
  ARMA.model<- l(to.ARMA(SS1.model),sub.sample)
  error <- abs(-2567.328011494376824 - ARMA.model$estimates$like[1])
  ok <- fuzz.large > error
  if (!ok) {if (is.na(max.error)) max.error <- error
            else max.error <- max(error, max.error)}
  all.ok <- all.ok & ok 
  if (verbose) 
    {if (ok) cat("ok\n")
     else cat("NOT CORRECT!\n")
    }

  if (verbose) cat("     ARMA model roots     ...")
  z <- roots(ARMA.model, fuzz=1e-4, by.poly=T)
  # the tolerance of this comparison had to be reduced because of changes
  #  from Splus 3.2 to Splus 3.3
  error <- abs(4.6786108692196117786 - sum(Mod(z)))
  ok <- 100*fuzz.large > error
  if (!ok) {if (is.na(max.error)) max.error <- error
            else max.error <- max(error, max.error)}
  all.ok <- all.ok & ok 
  if (verbose) 
    {if (ok) cat("ok\n")
     else cat("NOT CORRECT!\n")
    }

  if (summary) 
    {cat("All example tests completed")
     if (all.ok) cat(" ok\n")
     else    cat(", some failed! (max. error magnitude= ", max.error,")\n")
    }
  invisible(all.ok)
}

example.BOC.93.4.paper.tests <- function(example.data, example.raw.data, verbose=T, fuzz.small=1e-10)
{# Use as follows:
 #   example.BOC.93.4.paper.tests(example.BOC.93.4.data.all, 
 #      example.BOC.93.4.data.all.raw) 
 cat("This function reproduces some results from Bank of Canada Working Paper 93-4.\n")
   if (!example.verify.data(example.data, fuzz.small=fuzz.small))
     stop("example.data does not verify correctly. Testing stopped.")
   sub.sample <- list(
      input=window(input.data(example.data),end=c(1981,2)),
      output=window(output.data(example.data),end=c(1981,2)) )
   class(sub.sample) <- "TSdata"
   VAR.model <- est.VARX.ar(sub.sample, re.add.means=F)
   SS1.model <- l(balance.Mittnik(to.SS(VAR.model), n=9),sub.sample)
   g1 <- diag(1,9)
   g1[1:3,] <- SS1.model$model$H
   g1 <- solve(g1)
   g2 <- diag(1,9)
   g2[3,2:3] <- c(.1,-.1)
   g2[9,9] <- -1  # this is not really necessary but seems to have
                  #   happened in the paper
   example.gap.matrix <-g1 %*% g2
   SSgap.model <- l(gmap(example.gap.matrix,SS1.model),sub.sample)
   ARMA.model<- l(to.ARMA(SS1.model),sub.sample)

# the model parameters could be displayed at this point by:
#         display(VAR.model)   
#         display(SS1.model)   etc.

   cat("Likelihood of VAR model:                          ")
   print(VAR.model$estimates$like[1], digits=16)
   cat("Likelihood of Mittnik balanced state space model: ")
   print(SS1.model$estimates$like[1], digits=16)
   cat("Likelihood of state space `gap' model:            ")
   print(SSgap.model$estimates$like[1], digits=16)
   cat("Likelihood of ARMA model:                         ")
   print(ARMA.model$estimates$like[1], digits=16)
   cat("Remark: A small change has been made in the likelihood\n")
   cat("calculation since the version of the code used for\n")
   cat("calculating the results in Bank of Canada Working Paper 93-4.\n")
   cat("The new method is more robust to degenerate densities but gives\n")
   cat("a small difference in the likelihood value. (The value reported \n")
   cat(" was -2567.32801321424.      P.Gilbert.\n")
   
   cat("Stability of VAR model:\n")
   stability(VAR.model)
   cat("Stability of Mittnik balanced state space model:\n")
   stability(SS1.model)
   cat("Stability of state space `gap' model:\n")
   stability(SSgap.model)
   cat("Stability of ARMA model:\n")
   stability(ARMA.model)

   if(exists.graphics.device()) 
     {graph(VAR.model, Title="VAR model")
      cat("Remark: These are not advertised as best estimates. There is a bias.\n")
      cat("This estimation technique may be improved by setting some of the\n") 
      cat("options and other available estimation techniques work better.\n")
      cat("The example is intended primarily for illustrating the equivalence.\n")
      cat("press return to continue>");key<-scan(what="");cat("\n")
      graph(SS1.model, Title="Mittnik balanced state space model")
      cat("press return to continue>");key<-scan(what="");cat("\n")
      graph(SSgap.model, Title="State space `gap' model")
      cat("press return to continue>");key<-scan(what="");cat("\n")
      graph(ARMA.model,  Title="ARMA model")
      cat("press return to continue>");key<-scan(what="");cat("\n")
      model<- l(VAR.model,example.data)  # full sample
      example.show.ytoy.cpi(model,example.raw.data)  
        title(main="Predicted and actual CPI in terms of per cent change over 12 months")
      example.show.ytoy.cpi(model,example.raw.data, start=240)    
        title(main="Predicted and actual CPI in terms of per cent change over 12 months - ex post period")
     }
   
  invisible() 
}

example.show.ytoy.cpi <-function(model, raw.data, start = 1)
{
# plot cpi in year over year % change.
# prediction is relative to previous month's actual (raw.data)
# and % change is relative to actual.
# start is the starting point for plotting.
# base is the start value of the undif, un logged series.
        i <- 3 # cpi is the third variable
	base <- raw.data$output[1, i]
	pred <- model$estimates$pred[, i]
	y <- model$data$output[, i]
	y <- cumsum(c(log(base), y))
	pred <- c(log(base), pred)	# cumsum using pred relative to actual
	pred[2:length(pred)] <- pred[2:length(pred)] + y[1:(length(pred) - 1)]
	pred <- exp(pred)
	y <- exp(y)
	pred <- 100 * ((pred[13:length(pred)] - y[1:(length(y) - 12)])/y[1:(
		length(y) - 12)])
	y <- 100 * ((y[13:length(y)] - y[1:(length(y) - 12)])/y[1:(length(y) - 
		12)])
	plot(window(y, start=start),window(pred, start=start)) # tsplot
             invisible()
}

example.truncate.data <- function(d.all)
{ # truncate sample  to 240 periods.
  d <- list( input= input.data(d.all)[1:240,, drop=F], 
            output=output.data(d.all)[1:240,]) 
  dimnames(d$input) <- list(NULL, "R90")
  dimnames(d$output) <- list(NULL, c("M1","GDPl2", "CPI"))
  class(d) <-"TSdata"
  d
}


   example.BOC.93.4.data.all.raw<- 
       example.get.eg.raw.data(paste(DSE.HOME,"/data/eg1.dat", sep=""))
   #  old Users Guides still indicate eg1.DSE.data so ..
   eg1.DSE.data <- example.BOC.93.4.data.all.raw
   example.BOC.93.4.data.all <- 
       example.convert.eg.data(example.BOC.93.4.data.all.raw)
   example.BOC.93.4.data.trunc<-example.truncate.data(example.BOC.93.4.data.all)

#   Copyright 1993, 1994, 1995, 1996  Bank of Canada.
#   Copyright 1997 (June), Paul Gilbert.
#   Copyright 1997 (Aug.), Bank of Canada.
#   Copyright 1998, Bank of Canada.

#   The user of this software has the right to use, reproduce and distribute it.
#   Bank of Canada makes no warranties with respect to the software or its 
#   fitness for any particular purpose. The software is distributed by the Bank
#   of Canada and by Paul Gilbert solely on an "as is" basis. By using the  
#   software, user agrees to accept the entire risk of using this software.

################################################################################



# Once these functions and data have been loaded then an example 
# can be run by:
#        example.VAR.SVD(example.BOC.93.4.data.trunc)
# also recommended is:
#    example.verify.data(example.BOC.93.4.data.all) # prints some summary statistics

example.BOC.93.4.data.all <-example.convert.eg.data(example.BOC.93.4.data.all.raw)
# takes the log and difference of the output variables and puts the data in the variable example.BOC.93.4.data.all.



example.BOC.93.4.data.trunc <-example.truncate.data(example.BOC.93.4.data.all)

example.VAR.SVD <- function(d,d.all, d.all.raw)
 {V.1 <- est.VARX.ar(d) # estimates a VAR model using the truncated sample.

  l.V.1 <-l(V.1, d) # calculates the likelihood, one step ahead predictions, etc., and puts them in the variable l.V.1.

  cat("Likelihood and components for VAR model\n")
  # prints the likelihood value (with a breakdown for the 3 terms of the likelihood function).
  print(l.V.1$estimates$like, digits=16)

  #cat("Likelihood and components for VAR model\n")
  #l.V.1$estimates$like # also prints the value but not as many digits.
  # calculate the likelihood, one step ahead predictions, etc., based 
  #   on the full sample, and puts them in the variable o.V.1.
  o.V.1 <-l(V.1, d.all) 

  # convert the VAR model to a state space model balanced by Mittnik's technique.
  SS.V.1 <- to.SS(V.1) 

  # calculate the likelihood, one step ahead predictions, etc., based on 
  #   the truncated sample, and puts them in the variable l.SS.V.1.
  l.SS.V.1 <-l(SS.V.1, d) 

  cat("Likelihood and components for state space model\n")
  # print the likelihood value (with a breakdown for the 3 terms of 
  #     the likelihood function).
  print(l.SS.V.1$estimates$like,digits=16) 

  cat("Maximum difference in one-step predictions of VAR and state space model ")
  # calculate the difference of the absolute values of the predictions of 
  #      the two models.
  cat(max(abs(l.V.1$estimates$pred - l.SS.V.1$estimates$pred))) 
  cat("\n")

  cat("Exhibit 2. Mittnik reduction from VAR model: \n")
  M5.SS.V.1 <- reduction.Mittnik(SS.V.1,data=d, criterion="taic")  
   #If criterion is not specified the program prompts for a state dimension and 
   #returns that model. Results is put in the variable M5.SS.V.1.

  cat("Exhibit 3. Mittnik estimation lag=3: \n")
  M12.shift3 <- est.SS.Mittnik(d,max.lag=3, n=12)
  M12.shift3 <- reduction.Mittnik(M12.shift3, data=d, criterion="taic")  

  cat("Exhibit 4. Mittnik estimation lag=4: \n")
  M12.shift4 <- est.SS.Mittnik(d,max.lag=4, n=15)
  M12.shift4 <- reduction.Mittnik(M12.shift4, data=d, criterion="taic")  

  if(exists.graphics.device()) # eg.-OpenLook() or Suntools() 
     example.show.ytoy.cpi (o.V.1,d.all.raw,start=240)
  else  cat("Exhibit 8. graphic requires graphic device. \n")
  invisible()
}


#   Copyright 1993, 1994, 1995, 1996  Bank of Canada.
#   Copyright 1997 (June), Paul Gilbert.
#   Copyright 1997 (Aug.), Bank of Canada.
#   Copyright 1998, Bank of Canada.

#   The user of this software has the right to use, reproduce and distribute it.
#   Bank of Canada makes no warranties with respect to the software or its 
#   fitness for any particular purpose. The software is distributed by the Bank
#   of Canada and by Paul Gilbert solely on an "as is" basis. By using the  
#   software, user agrees to accept the entire risk of using this software.

################################################################################


# This files contains misc. routines for finding a vector parameter which
#  minimizes a function, and some utilities for gradient calculation and line 
#  search. The functions are setup so that the result can be used as the 
#  argument if the optimization is to be continued. 

############################################################################

#    functions for gradient calculation

############################################################################
numerical.grad <- function(func,x, eps=1e-12) {
#  very simple (crude) numerical approximation
  f <-func(x)
  df <-1:length(x)
  for (i in 1:length(x)) {
    dx <- x
    dx[i] <- dx[i] +eps 
    df[i] <- (func(dx)-f)/eps
   }
df
}

richardson.grad <- function(func, x, d=0.01, eps=1e-4, r=6, show.details=F){
# This function calculates a numerical approximation of the first
#   derivative of func at the point x. The calculation
#   is done by Richardson's extrapolation (see eg. G.R.Linfield and J.E.T.Penny
#   "Microcomputers in Numerical Analysis"). The method should be used if
#   accuracy, as opposed to speed, is important.
#
#  *  modified by Paul Gilbert from orginal code by XINGQIAO LIU.
# CALCULATES THE FIRST ORDER DERIVATIVE 
#     VECTOR using a Richardson  extrapolation.
#
#  GENERAL APPROACH
#     --  GIVEN THE FOLLOWING INITIAL VALUES:
#             INTERVAL VALUE D, NUMBER OF ITERATIONS R, AND
#             REDUCED FACTOR V.
#      - THE FIRST ORDER aproximation to the DERIVATIVE WITH RESPECT TO Xi IS
#
#           F'(Xi)={F(X1,...,Xi+D,...,Xn) - F(X1,...,Xi-D,...,Xn)}/(2*D)
#       
#     --  REPEAT r TIMES  with successively smaller D  and 
#          then apply Richardson extraplolation
#
#  INPUT
#       func    Name of the function.
#       x       The parameters of func.
#       d       Initial interval value (real) by default set to 0.01*x or
#               eps if x is 0.0.
#       r       The number of Richardson improvement iterations.
#       show.details    If T show intermediate results.
#  OUTPUT
#
#       The gradient vector.

  v <- 2               # reduction factor.
  n <- length(x)       # Integer, number of variables.
  a.mtr <- matrix(1, r, n) 
  b.mtr <- matrix(1, (r - 1), n)
#------------------------------------------------------------------------
# 1 Calculate the derivative formula given in 'GENERAL APPROACH' section.
#   --  The first order derivatives are stored in the matrix a.mtr[k,i], 
#        where the indexing variables k for rows(1 to r),  i for columns
#       (1 to n),  r is the number of iterations, and n is the number of
#       variables.
#-------------------------------------------------------------------------  

  h <- abs(d*x)+eps*(x==0.0)
  for(k in 1:r)  { # successively reduce h                
     for(i in 1:n)  {
         x1.vct <- x2.vct <- x
         x1.vct[i]  <- x[i] + h[i]
         x2.vct[i]  <- x[i] - h[i]
         if(k == 1) a.mtr[k,i] <- (func(x1.vct) - func(x2.vct))/(2*h[i])
         else{
           if(abs(a.mtr[(k-1),i])>1e-20)
                 # some functions are unstable near 0.0              
                 a.mtr[k,i] <- (func(x1.vct)-func(x2.vct))/(2*h[i])
            else  a.mtr[k, i] <- 0
          }
      }
     h <- h/v     # Reduced h by 1/v.
    }	
   if(show.details)  {
        cat("\n","first order approximations", "\n")		
        print(a.mtr, 12)
    }

#------------------------------------------------------------------------
# 1 Applying Richardson Extrapolation to improve the accuracy of 
#   the first and second order derivatives. The algorithm as follows:
#
#   --  For each column of the 1st and 2nd order derivatives matrix a.mtr,
#       say, A1, A2, ..., Ar, by Richardson Extrapolation, to calculate a
#       new sequence of approximations B1, B2, ..., Br used the formula
#
#          B(i) =( A(i+1)*4^m - A(i) ) / (4^m - 1) ,  i=1,2,...,r-m
#
#             N.B. This formula assumes v=2.
#
#   -- Initially m is taken as 1  and then the process is repeated 
#      restarting with the latest improved values and increasing the 
#      value of m by one each until m equals r-1
#
# 2 Display the improved derivatives for each
#   m from 1 to r-1 if the argument show.details=T.
#
# 3 Return the final improved  derivative vector.
#-------------------------------------------------------------------------

  for(m in 1:(r - 1)) {		
#     for(i in 1:(r - m)) b.mtr[i,]<- (a.mtr[(i+1),]*(4^m)-a.mtr[i,])/(4^m-1)
#     a.mtr<- b.mtr
     a.mtr<- (a.mtr[2:(r+1-m),]*(4^m)-a.mtr[1:(r-m),])/(4^m-1)
     if(show.details & m!=(r-1) )  {
        cat("\n","Richarson improvement group No. ", m, "\n")		
        print(a.mtr[1:(r-m),], 12)
      }
   }
a.mtr
}



############################################################################

#    functions for Davidon Fletcher Powell minimization
#      For more information about the algorithm 
#      see Numerical Recipes, William H. Press, B.P.Flannery, 
#      S.A. Teukolsky and W.T.Vetterling.      

############################################################################

dfpMin <- function(func, para, dfunc=numerical.grad, max.iter=500, ftol=1e-10, gtol=1e-3, eps=1e-4, line.search="brent", verbose=T) 
{ 
# func is the name of the function to be minimized. It must take a single, 
#    vector argument of the parameters for the minimization.
# dfunc is the name of a function which returns the gradient of func. It takes
#  two arguments, the first is the function name (func) and the second is the 
#  the vector of the parameters for the minimization. While the first 
#  arguement seems reduntant it makes the numerical approximation 
#  default convenient.
 
# para is a vector of parameters, or alternatively, a list as returned by this function, which can be used for continuation.
# max.iter is an integer indicating the maximum number of iterations.
# The value returned is a list (point.info) of:
#        1/   the function value, 
#        2/   the parameter values at the minimum (or at the last iteration if a min. is not reached), 
#        3/   the gradient
#        4/   the parameter hessian,
#        5/   and a logical indication of convergence (T=convergered),
#        6/   the search direction for the next step.

# The algorithm is modified from the original so that the grad and hessian are 
#    updated after convergence (which is a good idea if the hessian is used as 
#    the info. matrix).
# first define some utility functions:

dfpMin.guts <- function(func,dfunc,point.info, ftol=NULL, gtol=NULL, line.search=NULL, eps=NULL)
  {#  brent and mnbrak have been modified for vectors so linmin was trivial
   p <-point.info$parms
   xi<-point.info$search.direction
   if      (line.search=="nlmin") fret.p <-  nlmin.search(func,p, xi)
   else if (line.search=="brent") fret.p <-  brent(mnbrak(func,p,p+xi )) 
   fret<-fret.p[[1]]                 # new function value
   xi  <-fret.p[[2]] - p             # actual distance moved
   p   <-fret.p[[2]]                 # new point
   g   <- dfunc(func,p)
   dg  <- g - point.info$gradient        # difference between old and new grad
#   if (all(0 == dg)) dg <-rep(1e-20,length(dg)) # as unlikely as this may seem
   if (all(0 == xi)) xi <-rep(1e-20,length(xi))
   if (all(0 == g ))  g <-rep(1e-20,length(g))
   hessian<-point.info$hessian
   hdg <- c(hessian %*% dg)
   fac <- c(dg %*% xi)
   if (0 == fac) fac <- 1e-20   #  This kludge seems to work.
   else fac <- 1.0/fac          # fac can blow up if dg gets very small
   fae <- c(dg %*% hdg)         #  and fae can go to 0, causing hessian NaN.
   if (0 != fae) dg  <- c(fac*xi - hdg/fae)   # BFGS  modification to DFP
   else          dg  <- c(fac*xi)
   hessian <- hessian +  fac*outer(xi,xi)
   if (0!=fae) hessian<-hessian - outer(hdg,hdg)/fae + fae*outer(dg,dg)
   xi <-  - c(hessian %*% g)
##   xi <- xi - c(hessian %*% g)
   fp<-point.info$value
   if ((max(abs(g)) < ((gtol+abs(fret))*gtol)) & 
     ((2*abs(fret-fp)) <= (ftol*(abs(fret)+abs(fp)+eps)))) converged <- T
   else converged <- F
   list(value=fret,parms=p,gradient=g, hessian=hessian, 
        converged=converged,search.direction=xi)
} #end dfpMin.guts    

norm  <- function(x) {
 (sum(x^2))^0.5
}

nlmin.search <- function(func,p, x, damping=0.1)
{  nlmin.func <- function(r) {nlmin.func.name(nlmin.p+r*nlmin.x) }
   global.assign("nlmin.p"  , p)
   global.assign("nlmin.x"  , x)
   global.assign("nlmin.func.name"  , func)
   new.p <-p+x*nlmin(nlmin.func,damping)$x
   fx <- func(new.p)
   list(fx,new.p) 
}

mnbrak <- function(func,ax,bx) {
# Press et al p. 281
# from initial points ax and bx
# return three values fa,fb, and fc, and corresponding  points ax,bx,cx, which bracket the min of a function on a line.
# The orig. algor. shifts a,b, and c to a specific order before returning. This is accomplished by placement in the return list.
# The original algorithm has been modified to take vectors ax and bx and return vectors (but still does line searching).

   gold <-1.618034
   glimit <- 100.0
   tiny <- 1.0e-20

   fa <- func(ax)
   fb <- func(bx)
   if (fb > fa) {  # switch - remainder assumes  fa > fb
      dum <- ax
      ax  <- bx
      bx  <- dum
      dum <- fb
      fb  <- fa
      fa  <- dum
    }
   cx <- bx + gold*(bx-ax)
   fc<-func(cx)             # function evaluation 
   while( (fa+1e10) <= fc) {  # contract if function value is very large
      cx <- bx + 0.9*(cx-bx)
      fc<-func(cx) 
    }   
   while (fb >= fc) {
      r<-norm(bx-ax)*(fb-fc)   # modified for vector arguement  possible problem here
      q<-norm(bx-cx)*(fb-fa)   # modified for vector arguement
      if (q<r) { u<-bx-((bx-cx)*q-(bx-ax)*r)/(-2*max(abs(q-r),tiny)) }
      else     { u<-bx-((bx-cx)*q-(bx-ax)*r)/( 2*max(abs(q-r),tiny)) }
      ulim<-bx+glimit*(cx-bx)
      rad<-norm(cx-bx)             # modified for vector arguement
      if ((norm(u-bx) < rad) & (norm(u-cx) < rad) ) {   # u is between bx and cx
         fu<-func(u) 
         if (fu < fc) { return(list(func,fb,fu,fc,bx,u,cx))} # see note 1. 
         if (fu > fb) { return(list(func,fa,fb,fu,ax,bx,u))}
         u<-cx+gold*(cx-bx)
         fu<-func(u)
        } 
      else {                       # modified for vector arguement
         rad<-norm(cx-ulim)
         if ((norm(cx-u) < rad) & (norm(u-ulim) < rad) ){ # u between cx & ulim
            fu<-func(u)
            if (fu < fc) {
               bx<-cx
               cx<-u
               u<-cx+gold*(cx-bx)
               fb<-fc
               fc<-fu
               fu<-func(u)   
             }
          }   
      else {                     # modified for vector arguement
          rad<-norm(u-cx)
          if ((norm(u-ulim)<rad) & (norm(ulim-cx) <rad)) { # limit u
               u<-ulim
               fu<-func(u)
           }
      else {
           u<-cx+gold*(cx-bx)
           fu<-func(u)
       }}}
      ax <- bx
      bx <- cx
      cx <- u
      fa <- fb
      fb <- fc
      fc <- fu
   } # while
list(func,fa,fb,fc,ax,bx,cx)
}

brent <- function(arglist,tol=1e-10) {
# Press et al p.284    line search.
# for function func with bracket points ax, bx, cx (ie. ax & cx are either 
#  side of the min. and bx is between ax and cx and has a lower function value)
# return min  value (to tolerance tol) and location x.
# This is modified from the original algorithm to handle 
#     vector arguments ax,bx, and cx, which must be colinear.
# The vector arguements are converted to scalars using ax as the zero and cx-ax as a "unit basis ".
   func<-arglist[[1]]
   ax  <-arglist[[5]]
   bx  <-arglist[[6]]
   cx  <-arglist[[7]]

   itmax <-100
   cgold <- .3819660
   zeps<-1.0e-10
#   a <- min(ax,cx)
#   b <- max(ax,cx)
#   x <- w <- v <- bx
   a<-0.0                          # modified for vector arguement
   basis<-cx-ax                    # modified for vector arguement 
   unit<-b<-norm(cx-ax)             # modified for vector arguement
   x <- w <- v <- norm(bx-ax)     # modified for vector arguement
   e <-0.0
#   fw <- fv <- fx <- func(ax+basis*x/unit) # modified for vector arguement
   fw <- fv <- fx <- arglist[[3]]   # function value at bx is supplied
   for (iter in (1: itmax) ) {
      xm <- 0.5*(a+b)
      tol1<-tol*abs(x)+zeps
      tol2<-2.0*tol1
      if(abs(x-xm) <= (tol2-0.5*(b-a))) return(list(fx,ax+basis*x/unit)) 
               # min. found - return       # modified for vector arguement
      if( abs(e) > tol1) {
         r<-(x-w)*(fx-fv)
         q<-(x-v)*(fx-fw)
         p<-(x-v)*q-(x-w)*r
         q<-2.0*(q-r)
         if(q>0.0) p<--p
         q<-abs(q)
         etemp<-e
         e<-d
         if((abs(p) >= abs(0.5*q*etemp) ) | (p <=q*(a-x) ) | (p >= q*(b-x) ) ) {
            if (x >= xm) e <-a-x
            else e<-b-x
            d<-cgold*e
           }
          else {
             d<-p/q
             u<-x+d
             if ( ((u-a) <tol2) | ((b-u) <tol2))  {
                  if ((xm-x) < 0.0) d<--abs(tol1)
                  else d<-abs(tol1)
              }
           }
       }            
      else  {
         if (x >= xm) e <-a-x
         else e<-b-x
         d<-cgold*e
       }
      if (abs(d) >= tol1) { u<-x+d }
      else {
         if (d < 0.0) {u<-x-abs(d)}
         else         {u<-x+abs(d)}
       }
      fu<-func(ax+basis*u/unit)   #  modified for vector arguement
      if (fu <= fx) {
         if(u>=x) {a<-x}
         else     {b<-x}
         v<-w
         fv<-fw
         w<-x
         fw<-fx
         x<-u
         fx<-fu
        }
       else {
          if (u < x) {a<-u}
          else       {b<-u}
          if ((fu <=fw) | (w==x)) {
             v<-w
             w<-u
             fv<-fw
             fw<-fu
           }
          else {
             if ((fu <=fv) | (v==x) | (v==w)) {
                v<-u
                fv<-fu
              }
           }
        }
    } #for 
     print(" iteration limit exceeded in brent")
     return(list(fx,ax+basis*x/unit))         # modified for vector arguement
}



# begin dfpMin
global.assign("brent" , brent)
global.assign("mnbrak", mnbrak)
global.assign("norm"  , norm)
global.assign("nlmin.search"  , nlmin.search)
if ( ! is.list(para)) {
      v0 <-  func(para)
      if (verbose) cat("   iteration  0: ",v0,"\n")
      g<-dfunc(func,para)
      point.info <- list(value=v0,               # function value
                         parms=para,             # parameter values
                         gradient= g,            # gradient
                         hessian= diag(1,length(para)), # initial hessian = I
                         converged= FALSE,              # not converged
                         search.direction= -g)   # initial search direction = - grad
 }
else 
  {point.info <- para 
   if (verbose) cat("   iteration  0: ", point.info$value,"\n")
  }

iter     <-  0  
while ((iter < max.iter) & ( ! point.info$converged) ) { 
   iter     <-  iter + 1
   if (verbose) cat("   iteration " , iter)
   point.info <- dfpMin.guts(func,dfunc,point.info, ftol=ftol,gtol=gtol, line.search=line.search, eps=eps)
   if (verbose) cat(": " , point.info$value,"\n")
 } 
point.info
} #end dfpMin    


############################################################################

#    functions for non-linear simplex minimization

#    Procedure nlsimplex uses the Nelder-Mead algorithm to find a set of  
#     parameters that give a (local) minimum. Since we are trying to    
#     maximize the likelihood function we simply try to find the minimum *
#     of the negation of the likelihood function. For more information   
#     about the algorithm see Numerical Recipes, William H. Press, B.P.Flannery,
#     S.A. Teukolsky and W.T.Vetterling, p 289. 

############################################################################

nlsimplex <- function(func, pars, max.iter=500, verbose=T, report=100,
      alpha=1.0, beta=0.5, jamma=2.0, smallnum=-1e+13, ftol=1e-12) 
{# This version reflects away from the worst point (hi value)

 # Another version reflected toward the best point, but did not work very
 #    well and has been removed.

 # func is the name of the function to be minimized. It must take a single, 
 #     vector argument of the parameters for the minimization.
 # pars is a vector of parameters, or alternatively, a list as returned by this 
 # function, which can be used for continuation.
 # max.iter is an integer indicating the maximum number of iterations.
 # The value returned is a list of: the function values for points in the 
 #     simplex, arranged in increasing order, the array of simplex points, and 
 #     a logical indication of convergence (converged=T).

 #   alpha  reflection coefficient              
 #   beta   contraction coefficient    
 #   jamma  extrapolation coefficient
 #   ftol   function value tolerance for convergence   

 if (is.list(pars))
   {para  <- pars$simplex.parms
    y <- pars$simplex.values
    numpars <- dim(para)[2]
    mpts <- numpars +1
   }
 else 
  {numpars  <- length(pars)
   mpts     <-  1+numpars    # number of points in simplex         
   para <- t(matrix(c(pars,(matrix(pars,numpars,numpars)+diag(0.0000001*pars))),
                    numpars,mpts)) # initial simplex of parameters.
   y <- rep(NA,mpts)
   for (i in 1:mpts) y[i] <- func(c(para[i,])) #values for initial simplex
   }
               
 converged <- FALSE 
 iter <-  0  
 while ((iter <= max.iter) & (!converged) )
  {orderinc <- order(y)
   hi       <-  orderinc[mpts]   
   lo       <-  orderinc[1]   
   #   cat(" best: ", y[lo]," [ ", lo, "]"," worst: ", y[hi]," [ ", hi, "\n")
   nhi      <-  orderinc[mpts-1]           
   rtol     <-  2 * abs(y[hi] - y[lo]) / (abs(y[hi]) + abs(y[lo]) + 1e-25)
   if (rtol < ftol) converged <- TRUE 
   else
    {iter     <-  iter + 1
     if (verbose & (0 == iter %% report)) cat("iteration " , iter)
     pbar     <-  apply(para[-hi,, drop=F],2,sum) / numpars
     pr       <-  c( (1 + alpha) * pbar - alpha * para[hi,]) # reflect from hi
     ypr      <-  func(pr)   
     if (ypr <= y[lo])  
      {prr    <-  c( jamma * pr + (1 - jamma) * pbar )   # hi lo
       yprr <- func(prr)   
       if (yprr < y[lo])  
         {para[hi,]  <-  prr                
          y[hi]    <-  yprr                  
         }
        else                            
         {para[hi,]  <-  pr                 
          y[hi]    <-  ypr                   
         }                       
      } 
     else  
      {if (ypr >= y[nhi])   
        {if (ypr < y[hi] )        
          {para[hi,]  <-  pr                 
           y[hi]    <-  ypr                   
          }                            
         prr     <-  c( beta * para[hi,] + (1 - beta) * pbar )  # hi lo     
         yprr     <-  func(prr)  
         if ( yprr < y[hi])  
            {para[hi,]  <-  prr               
             y[hi]   <-  yprr                  
            }
         else 
           {para <-  0.5 * t(t(para) + para[lo,]) # repeats for columns
            for (i in seq(mpts)) if (i != lo) y[i] <- func(c(para[i,]))
           }                                         
          
        } 
      else                             
        {para[hi,]  <-  pr                
         y[hi]     <-  ypr               
        }
     }
 converged <- FALSE
 if(verbose & (0==iter %% report)) cat(" value ", y[1],"\n")
 } 
}

imin <- order(y)[1]
list(simplex.values=y, simplex.parms=para, 
     converged=converged, value=y[imin], parms=para[imin,])
} #end nlsimplex     

######################################################################


opt.function.tests <- function( verbose=T, synopsis=T,
                             fuzz.small=1e-9, fuzz.large=1e-4)
{# A short set of tests of the opt methods. 
 # dfpMin uses brent. nlmin could be tested too in Splus.

  simple.foo <- function(x)
    {th <- seq(length(x))
     B <- diag(th)
     c((x-th) %*% B %*% (x-th))
    }

  harder.foo <- function(x)
    {th <- seq(length(x))
     skew <- diag(1,length(x))
     skew[,1] <- -5^th
     B <- t(skew) %*% diag(10^th) %*% skew
     c((x-th) %*% B %*% (x-th))
    }

  all.ok <-  T
  if (synopsis & !verbose) cat("All opt tests ...")

  if (verbose) cat("opt test 1 ... ")
  z <- nlsimplex(simple.foo, rep(10,3), verbose=F, max.iter=100)
  z <- nlsimplex(simple.foo,    z ,     verbose=F)
  ok <- fuzz.small > max(abs(z$parms-1:3))      # para values
  ok <- ok & (fuzz.small > abs(z$value - 0.0) ) # fun value
  ok <- ok & z$converged                        # converged
  all.ok <- all.ok & ok 
  if (verbose) 
    {if (ok) cat("ok\n")
     else    cat("failed!\n")
    }

  if (verbose) cat("opt test 2 ... ")
  z <- dfpMin(simple.foo,rep(10,3), line.search="brent", max.iter=5,  verbose=F)
  z <- dfpMin(simple.foo,  z      , line.search="brent", 
               ftol=1e-10, gtol=1e-3, max.iter=20, verbose=F)
  ok <- fuzz.small > max(abs(z$parms-1:3))  # para values
  ok <- ok & (fuzz.small > abs(z$value-0) ) # fun value
  ok <- ok & z$converged
  all.ok <- ok
  if (verbose) 
    {if (ok) cat("ok\n")
     else    cat("failed!\n")
    }

  if (verbose) cat("opt test 2b... ")
  warning("hessian estimates seem to be double for some yet undetermined reason")
# if ftol is set higher the hessian gets worse !!!
  ok <- fuzz.large > max(abs(z$hessian - diag(1/(2*1:3))))  # hessian
  all.ok <- ok
  if (verbose) 
    {if (ok) cat("ok\n")
     else    {cat("failed!\n")}
    }

  if (verbose) cat("opt test 3 ... ")
  z <- dfpMin(simple.foo, rep(10,3), dfunc=numerical.grad, line.search="brent",
               ftol=1e-10, gtol=1e-3, max.iter=50, verbose=F)
  ok <- fuzz.small > max(abs(z$parms-1:3))  # para values
  ok <- ok & (fuzz.small > abs(z$value-0) ) # fun value
  ok <- ok & z$converged
  all.ok <- ok
  if (verbose) 
    {if (ok) cat("ok\n")
     else    cat("failed!\n")
    }

  if (verbose) cat("opt test 3b... ")
  ok <- fuzz.small > max(abs(z$hessian - diag(1/(2*1:3))))  # hessian
# first diagonal element of hessian seems to be going to correct value, second
#   and third are going to half.
  all.ok <- ok
  if (verbose) 
    {if (ok) cat("ok NOT USUAL\n")
     else    {cat("failed! as usual\n")}
    }


  if (verbose) cat("opt test 4 ... ")
  z <- dfpMin(harder.foo,rep(10,3), line.search="brent",
              gtol=0.05, max.iter=200, verbose=F)
  ok <- fuzz.large > max(abs(z$parms-1:3))  # para values
  ok <- ok & (fuzz.small > abs(z$value-0) ) # fun value
  ok <- ok & z$converged
  all.ok <- ok
  if (verbose) 
    {if (ok) cat("ok\n")
     else    cat("failed!\n")
    }

  if (verbose) cat("opt test 4b... ")
  th <- seq(length(1:3))
  skew <- diag(1,3)
  skew[,1] <- -5^th
  B <- t(skew) %*% diag(10^th) %*% skew
  ok <- fuzz.small > max(abs(z$hessian - solve(B)))  # hessian
  all.ok <- ok
  if (verbose) 
    {if (ok) cat("ok NOT USUAL\n")
     else    cat("failed! as usual\n")
    }

  if (verbose) cat("opt test 5 ... ")
  z <- dfpMin(harder.foo, rep(10,3), dfunc=richardson.grad, line.search="brent",
              max.iter=100, verbose=F)
  ok <- fuzz.small > max(abs(z$parms-1:3))  # para values
  ok <- ok & (fuzz.small > abs(z$value-0) ) # fun value
  ok <- ok & z$converged
  all.ok <- ok
  if (verbose) 
    {if (ok) cat("ok\n")
     else    {cat("failed!\n"); cat("val ",z$value," parms ",z$parms, "\n")}
    }

  if (verbose) cat("opt test 6 ... ")
  z <- nlsimplex(harder.foo, rep(10,3), max.iter=3000, ftol=1e-16, verbose=F)
  ok <- fuzz.small > max(abs(z$parms-1:3))      # para values
  ok <- ok & (fuzz.small > abs(z$value - 0.0) ) # fun value
  ok <- ok & z$converged                        # converged
  all.ok <- all.ok & ok 
  if (verbose) 
    {if (ok) cat("ok NOT USUAL\n")
     else    cat("failed! as usual\n")
    }


# now for and ARMA model based on example in user guide.
  ar<-array(c(1,.5,.3,0,.2,.1,0,.2,.05,1,.5,.3),c(3,2,2))
  ma<-array(c(1,.2,0,.1,0,0,1,.3),c(2,2,2))
  arma<-list(A=ar,B=ma,C=NULL)
  class(arma)<-c("ARMA","TSmodel")
  arma<-set.parameters(arma)
  if (is.S()) test.seed <- c(13,44,1,25,56,0,6,33,22,13,13,0)
  if (is.R()) test.seed <- c( 979, 1479, 1542)
  data <- simulate(arma, seed=test.seed, sampleT=500, sd=0.00001)
  like.value <- l(arma,data)$estimates$like[1]

  if (verbose) cat("opt test 7 ... ")
  global.assign("Obj.Func.ARGS", list(model=arma, data=data))
  on.exit("rm(Obj.Func.ARGS)")
  # following gives Warning: The cov. matrix is singular. Working on subspace.
  z <- dfpMin(like,parms(arma),line.search="brent", max.iter=500, verbose=T)
# this stalled about 22 with sd=0.0001 and  sampleT=100
  ok <- fuzz.small > max(abs(z$parms-parms(arma)))  # para values
  ok <- ok & (fuzz.small > abs(z$value-like.value) ) # fun value
  ok <- ok & z$converged
  all.ok <- ok
  if (verbose) 
    {if (ok) cat("ok\n")
     else    {cat("failed as usual!\n"); browser()}
    }

  if (verbose) cat("opt test 8 ... ")
  z <- dfpMin(like, parms(arma), dfunc=richardson.grad, line.search="brent",
              max.iter=500, verbose=T)
  ok <- fuzz.small > max(abs(z$parms-parms(arma)))  # para values
  ok <- ok & (fuzz.small > abs(z$value-like.value) ) # fun value
  ok <- ok & z$converged
  all.ok <- ok
  if (verbose) 
    {if (ok) cat("ok\n")
     else    {cat("failed as usual!\n"); browser()}
    }

  if (verbose) cat("opt test 9 ... ")
  z <- nlsimplex(like, parms(arma), max.iter=6000, ftol=1e-16, verbose=T)
  ok <- fuzz.small > max(abs(z$parms-parms(arma)))      # para values
  ok <- ok & (fuzz.small > abs(z$value - like.value) ) # fun value
  ok <- ok & z$converged                        # converged
  all.ok <- all.ok & ok 
  if (verbose) 
    {if (ok) cat("ok\n")
     else    {cat("failed as usual!\n"); browser()}
    }


  if (synopsis) 
    {if (verbose) cat("All opt tests completed")
     if (all.ok) cat(" ok\n")
     else    cat(", some failed!\n")
    }
  invisible(all.ok)
}

